#include "WeeklyViewWidget.h"

#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>
#include <QPixmap>
#include <QTimer>
#include <QKeyEvent>
#include <QTabletEvent>
#include "Configuration.h"
#include "DisplaySettings.h"
#include "Meeting.h"
#include "Room.h"
#include "MeetingRoomCombo.h"
#include "DigitalTimeDisplayWidget.h"
#include "ScheduleWidget.h"
#include "ToolBox.h"
#include "MeetingInfoDialog.h"

#include <QtDebug>

WeeklyViewWidget::WeeklyViewWidget( QDateTime aCurrentDateTime, Configuration *aConfiguration, QWidget *aParent ) :
		ObservedWidget( aParent ), iConfiguration( aConfiguration )
{
	// *****************************************
	//		Construct all the needed widgets
	QFont importantTextFont;
	importantTextFont.setBold( true );
	importantTextFont.setPointSize( 20 );

	QFont regularTextFont;
	regularTextFont.setBold( true );
	regularTextFont.setPointSize( 12 );

	iSettingsButton = new QPushButton;
	iSettingsButton->setIcon( QPixmap( ":button_settings" ) );
	iSettingsButton->setFixedWidth( 36 );
	iSettingsButton->setEnabled( false );
	connect( iSettingsButton, SIGNAL( clicked() ), this, SIGNAL( showSettingsView() ) );

	iCurrentDayLabel = ToolBox::createLabel( aCurrentDateTime.toString( iConfiguration->displaySettings()->dateFormat() ), regularTextFont );

	iRoomsCombo = new MeetingRoomCombo( iConfiguration->rooms(), this );
	iRoomsCombo->setCurrentRoom( iConfiguration->defaultRoom() );
	connect( iRoomsCombo, SIGNAL( observedEventDetected() ), this, SIGNAL( observedEventDetected() ) );

	iTimeDisplay = new DigitalTimeDisplayWidget( aCurrentDateTime.time(), iConfiguration->displaySettings()->timeFormat(), this );
	iTimeDisplay->setFrameVisible( false );
	iTimeDisplay->setFont( regularTextFont );
	connect( iTimeDisplay, SIGNAL( observedEventDetected() ), this, SIGNAL( observedEventDetected() ) );

	iSchedule = new ScheduleWidget( aCurrentDateTime, iConfiguration->displaySettings(), this );
	connect( iSchedule, SIGNAL( shownWeekChanged( int ) ), this, SLOT( shownWeekChanged( int ) ) );
	connect( iSchedule, SIGNAL( observedEventDetected() ), this, SIGNAL( observedEventDetected() ) );
	connect( iSchedule, SIGNAL( meetingActivated( Meeting* ) ), this, SIGNAL( meetingActivated( Meeting* ) ) );

	iPreviousWeekButton = new QPushButton( this );
	iPreviousWeekButton->setText( tr( "<<" ) );
	iPreviousWeekButton->setFixedWidth( 60 );
	connect( iPreviousWeekButton, SIGNAL( clicked() ), iSchedule, SLOT( showPreviousWeek() ) );

	iCurrentWeekButton = new QPushButton( this );
	iCurrentWeekButton->setFixedWidth( 100 );
	iCurrentWeekButton->setText( tr( "Current" ) );
	connect( iCurrentWeekButton, SIGNAL( clicked() ), iSchedule, SLOT( showCurrentWeek() ) );

	iNextWeekButton = new QPushButton( this );
	iNextWeekButton->setFixedWidth( 60 );
	iNextWeekButton->setText( tr( ">>" ) );
	connect( iNextWeekButton, SIGNAL( clicked() ), iSchedule, SLOT( showNextWeek() ) );

	// **********************************
	//		Create the view's layout
	QHBoxLayout *tableLayout = new QHBoxLayout;
	tableLayout->addWidget( iSchedule );

	QHBoxLayout *bottomLayout = new QHBoxLayout;
	bottomLayout->addWidget( iRoomsCombo );
	bottomLayout->addWidget( iTimeDisplay );
	bottomLayout->addWidget( iCurrentDayLabel );
	bottomLayout->addWidget( iPreviousWeekButton );
	bottomLayout->addWidget( iCurrentWeekButton );
	bottomLayout->addWidget( iNextWeekButton );
	bottomLayout->addWidget( iSettingsButton );

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout( tableLayout );
	mainLayout->addLayout( bottomLayout );
	setLayout( mainLayout );

	QPalette palette;
	palette.setColor( QPalette::Window, Qt::white );
	palette.setColor( QPalette::Foreground, Qt::darkGray );
	setPalette( palette );

	// ******************************************
	//		Handle all the signal connections
	// TODO : this solution if interaction monitoring is not elegant enough
	connect( iPreviousWeekButton, SIGNAL( clicked() ), this, SIGNAL( observedEventDetected() ) );
	connect( iCurrentWeekButton, SIGNAL( clicked() ), this, SIGNAL( observedEventDetected() ) );
	connect( iNextWeekButton, SIGNAL( clicked() ), this, SIGNAL( observedEventDetected() ) );
	connect( iRoomsCombo, SIGNAL( currentRoomChanged( Room * ) ), this, SIGNAL( observedEventDetected() ) );
	connect( iRoomsCombo, SIGNAL( currentIndexChanged( int ) ), this, SIGNAL( observedEventDetected() ) );
	// TODO: connect RoomCombo signals to change meetings data.
	connect( iRoomsCombo, SIGNAL( currentRoomChanged( Room * ) ), iSchedule, SLOT( clear() ) );
	connect( iRoomsCombo, SIGNAL( currentRoomChanged( Room * ) ), this, SIGNAL( currentRoomChanged( Room * ) ) );
	connect( iRoomsCombo, SIGNAL( currentRoomChanged( Room * ) ), iSchedule, SLOT( refresh() ) );
}

WeeklyViewWidget::~WeeklyViewWidget()
{
	if ( iRoomsCombo )
	{
		delete iRoomsCombo;
		iRoomsCombo = 0;
	}
	if ( iTimeDisplay )
	{
		delete iTimeDisplay;
		iTimeDisplay = 0;
	}
	if ( iSchedule )
	{
		delete iSchedule;
		iSchedule = 0;
	}
	if ( iCurrentDayLabel )
	{
		delete iCurrentDayLabel;
		iCurrentDayLabel = 0;
	}
	if ( iPreviousWeekButton )
	{
		delete iPreviousWeekButton;
		iPreviousWeekButton = 0;
	}
	if ( iCurrentWeekButton )
	{
		delete iCurrentWeekButton;
		iCurrentWeekButton = 0;
	}
	if ( iNextWeekButton )
	{
		delete iNextWeekButton;
		iNextWeekButton = 0;
	}
	if ( iSettingsButton )
	{
		delete iSettingsButton;
		iSettingsButton = 0;
	}
}

Meeting* WeeklyViewWidget::currentMeeting()
{
	return iSchedule->currentMeeting();
}

Room* WeeklyViewWidget::currentRoom()
{
	return iRoomsCombo->currentRoom();
}

void WeeklyViewWidget::setCurrentDateTime( QDateTime aCurrentDateTime )
{
	iCurrentDayLabel->setText( aCurrentDateTime.date().toString( iConfiguration->displaySettings()->dateFormat() ) );

	iTimeDisplay->setTime( aCurrentDateTime.time() );

	iSchedule->setCurrentDateTime( aCurrentDateTime );
}

void WeeklyViewWidget::insertMeeting( Meeting *aMeeting )
{
	iSchedule->insertMeeting( aMeeting );
}

void WeeklyViewWidget::deleteMeeting( Meeting *aMeeting )
{
	iSchedule->removeMeeting( aMeeting );
}

void WeeklyViewWidget::updateMeeting( Meeting *aMeeting )
{
	iSchedule->updateMeeting( aMeeting );
}

void WeeklyViewWidget::shownWeekChanged( int aWeekNumber )
{
	emit shownWeekChanged( iSchedule->beginningOfShownWeek() );
}

QDate WeeklyViewWidget::beginnigOfShownWeek()
{
	return iSchedule->beginningOfShownWeek();
}
