#include "SettingsView.h"

#include <QTabWidget>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QPushButton>
#include <QGroupBox>
#include <QLabel>
#include <QLineEdit>
#include <QRadioButton>
#include <QTimeEdit>
#include <QListView>
#include <QList>
#include <QTime>
#include <QIntValidator>
#include <QGridLayout>
#include <QCheckBox>
#include "Configuration.h"
#include "Room.h"
#include "DisplaySettings.h"
#include "ConnectionSettings.h"
#include "StartupSettings.h"

#include <QtDebug>

SettingsView::SettingsView( QWidget *aParent ) :
		ObservedWidget( aParent )
{
	qDebug() << "SettingsView::ctor invoked";
	// Prepare the tabbed view
	iTabWidget = new QTabWidget;

	iSettingsTab = initSettingsTab();
	iWeekViewTab = initWeekViewTab();
	iResourcesTab = initResourcesTab();
	iKioskModeTab = initKioskModeTab();

	iTabWidget->addTab( iSettingsTab, tr( "Settings" ) );
	iTabWidget->addTab( iWeekViewTab, tr( "Weekly View" ) );
	iTabWidget->addTab( iResourcesTab, tr( "Resources" ) );
	iTabWidget->addTab( iKioskModeTab, tr( "KIOSK Mode" ) );

	// Prepare the buttons and button layout
	QHBoxLayout *buttonLayout = new QHBoxLayout;
	iOkButton = new QPushButton;
	iOkButton->setText( tr( "OK" ) );
	iCancelButton = new QPushButton;
	iCancelButton->setText( tr( "Cancel" ) );
	buttonLayout->addWidget( iOkButton );
	buttonLayout->addWidget( iCancelButton );

	// Handle the main layout
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget( iTabWidget );
	mainLayout->addLayout( buttonLayout );

	setLayout( mainLayout );

	// Handle component connections
	connect( iOkButton, SIGNAL( pressed() ), this, SLOT( okClicked() ) );
	connect( iCancelButton, SIGNAL( pressed() ), this, SLOT( cancelClicked() ) );
}

SettingsView::~SettingsView()
{
	if ( iTabWidget != 0 )
	{
		delete iTabWidget;
		iTabWidget = 0;
	}
	if ( iOkButton != 0 )
	{
		delete iOkButton;
		iOkButton = 0;
	}
	if ( iCancelButton != 0 )
	{
		delete iCancelButton;
		iCancelButton = 0;
	}
	if ( iSettingsTab != 0 )
	{
		delete iSettingsTab;
		iSettingsTab = 0;
	}
	if ( iWeekViewTab != 0 )
	{
		delete iWeekViewTab;
		iWeekViewTab = 0;
	}
	if ( iResourcesTab != 0 )
	{
		delete iResourcesTab;
		iResourcesTab = 0;
	}
	if ( iKioskModeTab != 0 )
	{
		delete iKioskModeTab;
		iKioskModeTab = 0;
	}
	if ( iUserName != 0 )
	{
		delete iUserName;
		iUserName = 0;
	}
	if ( iPassword != 0 )
	{
		delete iPassword;
		iPassword = 0;
	}
	if ( iServerAddress != 0 )
	{
		delete iServerAddress;
		iServerAddress = 0;
	}
	if ( iDayStartTime != 0 )
	{
		delete iDayStartTime;
		iDayStartTime = 0;
	}
	if ( iDayEndTime != 0 )
	{
		delete iDayEndTime;
		iDayEndTime = 0;
	}
	if ( iFiveDays != 0 )
	{
		delete iFiveDays;
		iFiveDays = 0;
	}
	if ( iSevenDays != 0 )
	{
		delete iSevenDays;
		iSevenDays = 0;
	}
	if ( iRefreshInterval != 0 )
	{
		delete iRefreshInterval;
		iRefreshInterval = 0;
	}
	if ( iPowerSaveEnabled != 0 )
	{
		delete iPowerSaveEnabled;
		iPowerSaveEnabled = 0;
	}
	if ( iPowerSaveStartTime != 0 )
	{
		delete iPowerSaveStartTime;
		iPowerSaveStartTime = 0;
	}
	if ( iPowerSaveEndTime != 0 )
	{
		delete iPowerSaveEndTime;
		iPowerSaveEndTime = 0;
	}
}

QWidget *SettingsView::initSettingsTab()
{
	QWidget *widget = new QWidget( iTabWidget );

	// Prepare the widgets that are member variables
	iUserName = new QLineEdit;
	iPassword = new QLineEdit;
	iPassword->setEchoMode( QLineEdit::Password );
	iServerAddress = new QLineEdit;
	iRefreshInterval = new QLineEdit;
	QIntValidator *qiv = new QIntValidator( 0 );
	iRefreshInterval->setValidator( qiv );

	iUserName->setText( Configuration::instance()->connectionSettings()->username() );
	iPassword->setText( Configuration::instance()->connectionSettings()->password() );
	iServerAddress->setText( Configuration::instance()->connectionSettings()->serverUrl().toString() );
	QString refreshIntervalStr;
	refreshIntervalStr.setNum( Configuration::instance()->connectionSettings()->refreshInterval() );
	iRefreshInterval->setText( refreshIntervalStr );

	// Create the group boxes
	QGroupBox *userInformationGroup = new QGroupBox( tr( "User Information" ) );
	QGroupBox *serverInformationGroup = new QGroupBox( tr( "Server Information" ) );

	// Prepare the user infromation group box
	QGridLayout *ugl = new QGridLayout;
	QLabel *userNameLabel = new QLabel( tr( "Username:" ) );
	QLabel *passwordLabel = new QLabel( tr( "Password:" ) );

	ugl->addWidget( userNameLabel, 0, 0 );
	ugl->addWidget( iUserName, 0, 1 );
	ugl->addWidget( passwordLabel, 1, 0 );
	ugl->addWidget( iPassword, 1, 1 );

	userInformationGroup->setLayout( ugl );

	// Prepare the server information group box
	QGridLayout *sgl = new QGridLayout;
	QLabel *serverURLLabel = new QLabel( tr( "Server URL:" ) );
	QLabel *refreshLabel = new QLabel( tr( "Refresh interval" ) );
	QLabel *secondsLabel = new QLabel( tr( "seconds" ) );

	sgl->addWidget( serverURLLabel, 0, 0, 1, 2 );
	sgl->addWidget( iServerAddress, 0, 1 );
	sgl->addWidget( refreshLabel, 1, 0 );
	sgl->addWidget( iRefreshInterval, 1, 1 );
	sgl->addWidget( secondsLabel, 1, 2 );

	serverInformationGroup->setLayout( sgl );

	// Prepare and set the main layout
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget( userInformationGroup );
	mainLayout->addWidget( serverInformationGroup );

	widget->setLayout( mainLayout );

	return widget;
}

QWidget *SettingsView::initWeekViewTab()
{
	QWidget *widget = new QWidget( iTabWidget );

	// Prepare the member variable widgets
	iFiveDays = new QRadioButton( tr( "5" ) );
	iSevenDays = new QRadioButton( tr( "7" ) );
	iDayStartTime = new QTimeEdit;
	iDayEndTime = new QTimeEdit;

	if ( Configuration::instance()->displaySettings()->daysInSchedule() == DisplaySettings::WeekdaysInSchedule )
	{
		iFiveDays->setChecked( true );
		iSevenDays->setChecked( false );
	}
	else
	{
		iFiveDays->setChecked( false );
		iSevenDays->setChecked( true );
	}
	iDayStartTime->setTime( Configuration::instance()->displaySettings()->dayStartsAt() );
	iDayEndTime->setTime( Configuration::instance()->displaySettings()->dayEndsAt() );

	// Create group box and the grid layout
	QGroupBox *weeklyInformation = new QGroupBox( tr( "Weekly View" ) );
	QGridLayout *wgl = new QGridLayout;

	// Prepare the number of days row
	QLabel *daysLabel = new QLabel( tr( "Days:" ) );

	wgl->addWidget( daysLabel, 0, 0 );
	wgl->addWidget( iFiveDays, 0, 1 );
	wgl->addWidget( iSevenDays, 0, 2 );

	// Preare the day starts row
	QLabel *dayStartsLabel = new QLabel( tr( "Day starts:" ) );

	wgl->addWidget( dayStartsLabel, 1, 0 );
	wgl->addWidget( iDayStartTime, 1, 1, 1, 2 );

	// Prepare the day ends row
	QLabel *dayEndsLabel = new QLabel( tr( "Day ends:" ) );

	wgl->addWidget( dayEndsLabel, 2, 0 );
	wgl->addWidget( iDayEndTime, 2, 1, 1, 2 );

	weeklyInformation->setLayout( wgl );

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget( weeklyInformation );

	widget->setLayout( mainLayout );

	return widget;
}

QWidget *SettingsView::initResourcesTab()
{
	QWidget *widget = new QWidget( iTabWidget );

	QHBoxLayout *mainLayout = new QHBoxLayout;

	// Available resources
	QVBoxLayout *availableResourcesLayout = new QVBoxLayout;
	QLabel *availableResourcesLabel = new QLabel( tr( "Available Resources:" ) );
	QListView *availableResourcesList = new QListView;

	// Fill the list
	QList<Room*> rooms = Configuration::instance()->rooms();
	for ( int i = 0; i < rooms.count(); i++ )
	{
		Room *tmp_room = ( Room * ) rooms.at( i );
		qDebug() << "Room: " << tmp_room->name();
	}

	availableResourcesLayout->addWidget( availableResourcesLabel );
	availableResourcesLayout->addWidget( availableResourcesList );

	// Selected resources
	QVBoxLayout *selectedResourcesLayout = new QVBoxLayout;
	QLabel *selectedResourcesLabel = new QLabel( tr( "Selected Resources:" ) );
	QListView *selectedResourcesList = new QListView;

	selectedResourcesLayout->addWidget( selectedResourcesLabel );
	selectedResourcesLayout->addWidget( selectedResourcesList );

	// Button lauout
	QVBoxLayout *controlButtonsLayout = new QVBoxLayout;
	QPushButton *addButton = new QPushButton( tr( "->" ) );
	QPushButton *removeButton = new QPushButton( tr( "<-" ) );
	controlButtonsLayout->addWidget( addButton );
	controlButtonsLayout->addWidget( removeButton );

	// Prepare main layout
	mainLayout->addLayout( availableResourcesLayout );
	mainLayout->addLayout( controlButtonsLayout );
	mainLayout->addLayout( selectedResourcesLayout );

	widget->setLayout( mainLayout );

	return widget;
}

QWidget *SettingsView::initKioskModeTab()
{
	QWidget *widget = new QWidget( iTabWidget );

	// Prepare member variable widgets
	iPowerSaveEnabled = new QCheckBox( tr( "Power save enabled" ) );
	iPowerSaveStartTime = new QTimeEdit;
	iPowerSaveEndTime = new QTimeEdit;

	if ( Configuration::instance()->startupSettings()->isPowersavingEnabled() )
	{
		iPowerSaveEnabled->setChecked( true );
	}
	else
	{
		iPowerSaveEnabled->setChecked( false );
	}
	iPowerSaveStartTime->setTime( Configuration::instance()->startupSettings()->turnOnAt() );
	iPowerSaveEndTime->setTime( Configuration::instance()->startupSettings()->turnOffAt() );

	// Prepare the admin password box
	QGroupBox *adminPasswordGroup = new QGroupBox( tr( "Admin Password" ) );
	QLabel *oldPwdLabel = new QLabel( tr( "Old password:" ) );
	QLabel *newPwdLabel = new QLabel( tr( "New password:" ) );
	QLabel *confirmPwdLabel = new QLabel( tr( "Confirm password:" ) );
	QPushButton *applyPwdButton = new QPushButton( tr( "Apply" ) );

	QLineEdit *oldPwdEdit = new QLineEdit;
	QLineEdit *newPwdEdit = new QLineEdit;
	QLineEdit *confirmPwdEdit = new QLineEdit;

	oldPwdEdit->setEchoMode( QLineEdit::Password );
	newPwdEdit->setEchoMode( QLineEdit::Password );
	confirmPwdEdit->setEchoMode( QLineEdit::Password );

	QGridLayout *agl = new QGridLayout;

	agl->addWidget( oldPwdLabel, 0, 0 );
	agl->addWidget( oldPwdEdit, 0, 1 );
	agl->addWidget( newPwdLabel, 1, 0 );
	agl->addWidget( newPwdEdit, 1, 1 );
	agl->addWidget( confirmPwdLabel, 2, 0 );
	agl->addWidget( confirmPwdEdit, 2, 1 );
	agl->addWidget( applyPwdButton, 3, 0, 1, 2, Qt::AlignRight );

	adminPasswordGroup->setLayout( agl );

	// Prepare the power save options
	QGroupBox *powerSaveGroup = new QGroupBox( tr( "Power Save" ) );
	QLabel *switchedOnLabel = new QLabel( tr( "Switched on from:" ) );
	QLabel *toLabel = new QLabel( tr( "to" ) );
	QGridLayout *psgl = new QGridLayout;

	psgl->addWidget( iPowerSaveEnabled, 0, 0, 1, 4, Qt::AlignLeft );
	psgl->addWidget( switchedOnLabel, 1, 0 );
	psgl->addWidget( iPowerSaveStartTime, 1, 1 );
	psgl->addWidget( toLabel, 1, 2 );
	psgl->addWidget( iPowerSaveEndTime, 1, 3 );

	powerSaveGroup->setLayout( psgl );

	// Prepare the main layout
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget( adminPasswordGroup );
	mainLayout->addWidget( powerSaveGroup );

	widget->setLayout( mainLayout );

	return widget;
}

void SettingsView::okClicked()
{
	qDebug() << "[SettingsView::okClicked] <Invoked>";

	// Collect the configration data
	QTime calendarStart = iDayStartTime->time();
	QTime calendarEnd = iDayEndTime->time();
	QTime powerSaveStart = iPowerSaveStartTime->time();
	QTime powerSaveEnd = iPowerSaveEndTime->time();

	QString userName = iUserName->text();
	QString password = iPassword->text();
	QString serverAddress = iServerAddress->text();
	QString refreshInterval = iRefreshInterval->text();

	bool fiveDays = iFiveDays->isChecked();
	bool sevenDays = iSevenDays->isChecked();
	bool powerSaveEnabled = iPowerSaveEnabled->isChecked();

	// TODO : Set the values to configuration and save it

	close();
}

void SettingsView::cancelClicked()
{
	qDebug() << "[SettingsView::cancelClicked] <Invoked>";
	close();
}
