#ifndef ROOMSTATUSINDICATORWIDGET_H_
#define ROOMSTATUSINDICATORWIDGET_H_

#include "ObservedWidget.h"
#include <QTime>
#include <QKeyEvent>
#include <QTabletEvent>

#include "Room.h"

class QLabel;
class QVBoxLayout;
class TimeDisplayWidget;

//! UserInterface class. Indicates if the default meeting room is busy or not.
/*!
 * UserInterface class. Indicates if the default meeting room is busy or not. This widget
 * appears on the screen if there was no user interaction with the device for a ceratin period of time,
 * and disappears if there is any. Its function is to behave like a screen saver on one hand, and
 * to provide details about the current availability on the other hand.
 */
class RoomStatusIndicatorWidget : public ObservedWidget
{
	Q_OBJECT

public:
	//! Contructor.
	/*!
	 * Constructor to initialize RoomStatusIndicatorWidget instances.
	 * \param aDefaultRoom The room which's status is indicated by the current instance.
	 * \param aStatus The current status of the room.
	 * \param aUntil Time until the availability information is valid.
	 * \param aTimeFormat Time format string.
	 * \param aParent Pointer to the parent widget. Optional.
	 *
	 */
	RoomStatusIndicatorWidget( Room *aDefaultRoom, Room::Status aStatus, QTime aUntil, QString aTimeFormat, QWidget *aParent = 0 );
	//! Destructor.
	virtual ~RoomStatusIndicatorWidget();

public slots:
	//! Slot. Sets current time.
	/*!
	 * Slots. Sets current time on the widget. It is used to provide up-to-date time for the widget's
	 * TimeDisplayWidget.
	 * \param aCurrentTime The current time.
	 */
	void setCurrentTime( QTime aCurrentTime );
	//! Slot. Used to indicate changes in the status of the default room.
	/*!
	 * Slot. Used to indicate changes in the status of the default room. If the specified until time equals
	 * with the end of the day, i.e. there are no further meetings today or the room is busy until the end
	 * of the day, then not the until time is shown but a human readable informative message instead.
	 * \param aStatus The new status of the meeting room.
	 * \param aUntil The new time until the specified status is valid.
	 */
	void statusChanged( const Room::Status aStatus, const QTime aUntil );

private:
	//! Translates the status into human readable text.
	/*!
	 * Translates the status into human readable text.
	 * \param aStatus The status to be translated.
	 * \return The string translation of the status.
	 */
	QString statusToText( const Room::Status aStatus );
	//! Creates a palette used to indicate background color based on the specified status.
	/*!
	 * Creates a palette used to indicate background color based on the specified status.
	 * \param aStatus The status.
	 * \return The palette.
	 */
	QPalette createPalette( Room::Status aStatus );

private:
	QLabel *iDefaultRoomLabel;
	QLabel *iStatusLabel;
	QLabel *iUntilTextLabel;
	TimeDisplayWidget *iTimeDisplay;
	QString iTimeFormat;

	static QTime endOfTheDay;

};

#endif /*ROOMSTATUSINDICATORWIDGET_H_*/
