#include "RoomStatusIndicatorWidget.h"

#include <QLabel>
#include <QFont>
#include <QVBoxLayout>
#include "DigitalTimeDisplayWidget.h"
#include "ToolBox.h"

#include <QtDebug>

QTime RoomStatusIndicatorWidget::endOfTheDay = QTime( 23, 59, 0, 0 );

RoomStatusIndicatorWidget::RoomStatusIndicatorWidget( Room *aDefaultRoom, Room::Status aStatus, QTime aUntil, QString aTimeFormat, QWidget *aParent ) :
		ObservedWidget( aParent ), iTimeFormat( aTimeFormat )
{
	QFont importantTextFont;
	//importantTextFont.setBold( true );
	importantTextFont.setPointSize( 20 );

	QFont regularTextFont;
	//regularTextFont.setBold( true );
	regularTextFont.setPointSize( 12 );

	// display for current time
	// Note: the time display receives current time info from Engine::clock()
	iTimeDisplay = new DigitalTimeDisplayWidget( QTime::currentTime(), iTimeFormat, this );
	iTimeDisplay->setFrameVisible( false );
	iTimeDisplay->setSize( 250, 120 );

	// Pegasus
	iDefaultRoomLabel = ToolBox::createLabel( aDefaultRoom->name(), importantTextFont );
	iDefaultRoomLabel->setAlignment( Qt::AlignHCenter );
	iDefaultRoomLabel->setStyleSheet( "background-color: transparent" );
	// is busy
	iStatusLabel = ToolBox::createLabel( tr( "is %1" ).arg( statusToText( aStatus ) ), importantTextFont );
	iStatusLabel->setAlignment( Qt::AlignHCenter );
	iStatusLabel->setStyleSheet( "background-color: transparent" );

	// until 13:22
	iUntilTextLabel = ToolBox::createLabel( tr( "until %1" ).arg( aUntil.toString( iTimeFormat ) ), importantTextFont );
	iUntilTextLabel->setAlignment( Qt::AlignHCenter );
	iUntilTextLabel->setStyleSheet( "background-color: transparent" );

	QVBoxLayout *topLayout = new QVBoxLayout;
	topLayout->addStretch();
	topLayout->addWidget( iTimeDisplay );
	topLayout->addWidget( iDefaultRoomLabel );
	topLayout->addWidget( iStatusLabel );
	topLayout->addWidget( iUntilTextLabel );
	topLayout->addStretch();

	QHBoxLayout *mainLayout = new QHBoxLayout;
	mainLayout->addLayout( topLayout );
	mainLayout->addStretch();
	mainLayout->setMargin( 65 );
	setLayout( mainLayout );

	statusChanged( aStatus, aUntil );

	setFocusPolicy( Qt::StrongFocus );
	setEnabled( true ); // enable mouse & key events
}

RoomStatusIndicatorWidget::~RoomStatusIndicatorWidget()
{
	delete iTimeDisplay;
	iTimeDisplay = 0;
}

QString RoomStatusIndicatorWidget::statusToText( const Room::Status aStatus )
{
	return ( aStatus == Room::BusyStatus ) ? tr( "busy" ) : tr( "free" );
}

QPalette RoomStatusIndicatorWidget::createPalette( Room::Status aStatus )
{
	QBrush brush;
	brush.setTexture( aStatus == Room::BusyStatus ? QPixmap( ":roomstatus_busy" ) : QPixmap( ":roomstatus_free" ) );
	QPalette palette;
	palette.setBrush( QPalette::Window, brush );
	return palette;
}

void RoomStatusIndicatorWidget::setCurrentTime( QTime aCurrentTime )
{
	iTimeDisplay->setTime( aCurrentTime );
}

void RoomStatusIndicatorWidget::statusChanged( const Room::Status aStatus, const QTime aUntil )
{
	iStatusLabel->setText( tr( "is %1" ).arg( statusToText( aStatus ) ) );
	if ( aUntil == RoomStatusIndicatorWidget::endOfTheDay )
	{
		iUntilTextLabel->setText( tr( "whole day." ) );
	}
	else
	{
		iUntilTextLabel->setText( tr( "until %1" ).arg( aUntil.toString( iTimeFormat ) ) );
	}
	setPalette( createPalette( aStatus ) );
}
