#ifndef POPUPMESSAGEBOX_H_
#define POPUPMESSAGEBOX_H_

#include <QDialog>

class QLabel;

//! UserInterface class. Responsible for poping-up information messages on the screen.
/*!
 * UserInterface class. Responsible for poping-up information messages on the screen. The class is used
 * through static methods, just like the QMessageBox in the standard Qt library.
 */
class PopUpMessageBox : public QDialog
{
	Q_OBJECT

public:
	//! Enumeration of dialog buttons.
	/*!
	 * Predefined buttons for the pop up dialogs.
	 */
	enum Button
	{
		NoButtonAtAll, /*!< There is no button. Dialog pops up for the specified interval, then automatically closes. */
		OkButton, /*!< OK button on the right. */
		YesNoButton /*!< Yes and No buttons. */
	};
	//! Enumeration of possible return types after the specified Key was pressed.
	/*!
	 * Predefined return types after the specified Key was pressed.
	 * \sa PopUpMessageBox::Key.
	 */
	enum ReturnType
	{
		YesReturnType, /*!< Yes was pressed. */
		NoReturnType, /*!< No was pressed. */
		OkReturnType /*!< OK was pressed. */
	};

signals:
	//! Signals if user presses a key available for the dialog.
	/*!
	 * The signal is emitted if user presses a key available for the dialog.
	 * \param aReturnType The return type based on the key.
	 * \sa PopUpMessageBox::ReturnType.
	 */
	void buttonPressed( PopUpMessageBox::ReturnType aReturnType );

private slots:
	void yesButtonPressed();
	void noButtonPressed();
	void okButtonPressed();

private:
	//! Constructor.
	/*!
	 * Constructor to initialize a PopUpMessageBox instance.
	 * \param aParent Optional. The parent object.
	 * \param aTitle Optional. The title of the dialog.
	 * \param aText Optional. The message the dialog supposed to show.
	 * \param aKey Optional. The keys which are accessible when the dialog is active. There is no key by default.
	 * \sa PopUpMessage::Key.
	 */
	PopUpMessageBox( QWidget *aParent = 0, const QString &aTitle = 0, const QString &aText = 0, Button aButton = NoButtonAtAll );
	//! Destructor.
	virtual ~PopUpMessageBox();

	//! Gets the title.
	/*!
	 * Getter method to provide the title attribute of the object.
	 * \return The title as a QString.
	 */
	QString getTitle();
	//! Gets the message text.
	/*!
	 * Getter method to provide the message text of the dialog.
	 * \return The message as a QString.
	 */
	QString getMessage();
	//! Sets the icon for the dialog.
	/*!
	 * Sets the icon for the dialog.
	 * \param aPath The full path of the icon.
	 * \return TRUE if setting of icon was successful; otherwise, FALSE.
	 */
	bool setIcon( const QString &aPath );
	//! Sets the title.
	/*!
	 * Sets the title of the current object.
	 * \param aTitle The new title.
	 */
	void setTitle( const QString &aTitle );
	//! Sets the message text.
	/*!
	 * Sets the message text of the current object.
	 * \param aText The new text.
	 */
	void setMessage( const QString &aText );
	//! Sets the interval of being popped up.
	/*!
	 * Sets the interval of the current object until when it is popped up.
	 * \param aInterval The new interval.
	 */
	void setInterval( int aInterval );
	//! Pops the dialog up.
	/*!
	 * Pops the dialog up for the specified time interval if no key is specified; otherwise, until there is
	 * a key (actually softkey button) pressed.
	 * \sa PopUpMessageBox::Key.
	 */
	void popUp();

public:
	//! Creates a new PopUpMessageBox instance.
	/*!
	 * Creates a new 'Cancel' PopUpMessage instance without buttons.
	 * \param aParent The parent object.
	 * \param aText Optional. The message text.
	 * \return The instance which was created.
	 * \sa popUp().
	 */
	static PopUpMessageBox * cancel( QWidget *aParent, const QString &aText = 0 );
	//! Creates a new PopUpMessageBox instance.
	/*!
	 * Creates a new 'Error' PopUpMessage instance without buttons.
	 * \param aParent The parent object.
	 * \param aText Optional. The message text.
	 * \return The instance which was created.
	 * \sa popUp().
	 */
	static PopUpMessageBox * error( QWidget *aParent, const QString &aText = 0 );
	//! Creates a new PopUpMessageBox instance.
	/*!
	 * Creates a new 'Information' PopUpMessage instance without buttons.
	 * \param aParent The parent object.
	 * \param aText Optional. The message text.
	 * \return The instance which was created.
	 * \sa popUp().
	 */
	static PopUpMessageBox * information( QWidget *aParent, const QString &aText = 0 );
	//! Creates a new PopUpMessageBox instance.
	/*!
	 * Creates a new 'OK' PopUpMessage instance without buttons.
	 * \param aParent The parent object.
	 * \param aText Optional. The message text.
	 * \return The instance which was created.
	 * \sa popUp().
	 */
	static PopUpMessageBox * ok( QWidget *aParent, const QString &aText = 0 );
	//! Creates a new PopUpMessageBox instance.
	/*!
	 * Creates a new 'Question' PopUpMessage instance with 'Yes' and 'No' buttons.
	 * \param aParent The parent object.
	 * \param aText Optional. The message text.
	 * \return The instance which was created.
	 * \sa popUp().
	 */
	static PopUpMessageBox * question( QWidget *aParent, const QString &aText = 0 );
	//! Creates a new PopUpMessageBox instance.
	/*!
	 * Creates a new 'Warning' PopUpMessage instance without buttons.
	 * \param aParent The parent object.
	 * \param aText Optional. The message text.
	 * \return The instance which was created.
	 * \sa popUp().
	 */
	static PopUpMessageBox * warning( QWidget *aParent, const QString &aText = 0 );

public:
	//! The default pop-up interval in milliseconds.
	static const int DefaultInterval = 2500; // ms

private:
	QLabel *iIconLabel;
	QLabel *iMessageLabel;
	QTimer *iTimer;
	Button iButton;

	static const int sDialogWidth = 175;
	static const int sDialogHeight = 125;
	static const int sIconSize = 25;

};


#endif /*POPUPMESSAGEBOX_H_*/
