#include "PopUpMessageBox.h"
#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>
#include <QTimer>
#include <QImage>
#include <QApplication>

PopUpMessageBox::PopUpMessageBox( QWidget *aParent, const QString &aTitle, const QString &aText, Button aButton ) :
		QDialog( aParent )
{
	setWindowTitle( aTitle );
	setModal( true );

	iIconLabel = new QLabel;
	iIconLabel->setBackgroundRole( QPalette::HighlightedText );
	iIconLabel->setMaximumSize( PopUpMessageBox::sIconSize, PopUpMessageBox::sIconSize );
	//iIconLabel->setScaledContents( true );

	iMessageLabel = new QLabel;
	iMessageLabel->setWordWrap( true );
	setMessage( aText );

	QHBoxLayout *topLayout = new QHBoxLayout;
	topLayout->addWidget( iIconLabel );
	topLayout->addSpacing( 5 );
	topLayout->addWidget( iMessageLabel, 10 );

	QVBoxLayout *layout = new QVBoxLayout;
	layout->addLayout( topLayout );

	iButton = aButton;
	switch ( iButton )
	{
		case OkButton:
		{
			QPushButton *okButton = new QPushButton;
			okButton->setText( tr( "OK" ) );
			connect( okButton, SIGNAL( pressed() ), this, SLOT( okButtonPressed() ) );

			QHBoxLayout *buttonLayout = new QHBoxLayout;
			buttonLayout->addStretch();
			buttonLayout->addWidget( okButton );
			layout->addSpacing( 5 );
			layout->addLayout( buttonLayout );
			break;
		}
		case YesNoButton:
		{
			QPushButton *yesButton = new QPushButton;
			yesButton->setText( tr( "Yes" ) );
			connect( yesButton, SIGNAL( pressed() ), this, SLOT( yesButtonPressed() ) );

			QPushButton *noButton = new QPushButton;
			noButton->setText( tr( "No" ) );
			connect( noButton, SIGNAL( pressed() ), this, SLOT( noButtonPressed() ) );

			QHBoxLayout *buttonLayout = new QHBoxLayout;
			buttonLayout->addStretch();
			buttonLayout->addWidget( yesButton );
			buttonLayout->addWidget( noButton );
			layout->addSpacing( 5 );
			layout->addLayout( buttonLayout );
		}
		default: // NoButtonAtAll
		{
			iTimer = new QTimer;
			setInterval( PopUpMessageBox::DefaultInterval );
			connect( iTimer, SIGNAL( timeout() ), this, SLOT( close() ) );
			break;
		}
	}

	setLayout( layout );

	resize( PopUpMessageBox::sDialogWidth, PopUpMessageBox::sDialogHeight );
}

PopUpMessageBox::~PopUpMessageBox()
{
}

QString PopUpMessageBox::getTitle()
{
	return windowTitle();
}

QString PopUpMessageBox::getMessage()
{
	return iMessageLabel->text();
}

bool PopUpMessageBox::setIcon( const QString &aPath )
{
	iIconLabel->clear();
	QImage image( aPath );
	if ( image.isNull() )
	{
		return false;
	}
	iIconLabel->setPixmap( QPixmap::fromImage( image ) );
	return true;
}

void PopUpMessageBox::setTitle( const QString &aTitle )
{
	setWindowTitle( aTitle );
}

void PopUpMessageBox::setMessage( const QString &aMsg )
{
	iMessageLabel->setText( aMsg );
}

void PopUpMessageBox::setInterval( int aInternal )
{
	iTimer->setInterval( aInternal );
}

void PopUpMessageBox::popUp()
{
	show();
	if ( iButton == NoButtonAtAll )
	{
		iTimer->start();
	}
}

void PopUpMessageBox::yesButtonPressed()
{
	emit buttonPressed( PopUpMessageBox::YesReturnType );
	close();
}

void PopUpMessageBox::noButtonPressed()
{
	emit buttonPressed( PopUpMessageBox::NoReturnType );
	close();
}

void PopUpMessageBox::okButtonPressed()
{
	emit buttonPressed( PopUpMessageBox::OkReturnType );
	close();
}

PopUpMessageBox * PopUpMessageBox::cancel( QWidget *aParent, const QString &aText )
{
	PopUpMessageBox* msg = new PopUpMessageBox( aParent, tr( "Cancel" ), aText, NoButtonAtAll );
	msg->setIcon( ":/popup_cancel" );
	msg->popUp();
	QApplication::beep();
	return msg;
}

PopUpMessageBox * PopUpMessageBox::error( QWidget *aParent, const QString &aText )
{
	PopUpMessageBox* msg = new PopUpMessageBox( aParent, tr( "Error" ), aText, NoButtonAtAll );
	msg->setIcon( ":/popup_error" );
	msg->popUp();
	QApplication::beep();
	return msg;
}

PopUpMessageBox * PopUpMessageBox::information( QWidget *aParent, const QString &aText )
{
	PopUpMessageBox* msg = new PopUpMessageBox( aParent, tr( "Information" ), aText, NoButtonAtAll );
	msg->setIcon( ":/popup_information" );
	msg->popUp();
	return msg;
}

PopUpMessageBox * PopUpMessageBox::ok( QWidget *aParent, const QString &aText )
{
	PopUpMessageBox* msg = new PopUpMessageBox( aParent, tr( "OK" ), aText, OkButton );
	msg->setIcon( ":/popup_ok" );
	msg->popUp();
	return msg;
}

PopUpMessageBox * PopUpMessageBox::warning( QWidget *aParent, const QString &aText )
{
	PopUpMessageBox* msg = new PopUpMessageBox( aParent, tr( "Warning" ), aText, NoButtonAtAll );
	msg->setIcon( ":/popup_warning" );
	msg->popUp();
	QApplication::beep();
	return msg;
}

PopUpMessageBox * PopUpMessageBox::question( QWidget *aParent, const QString &aText )
{
	PopUpMessageBox* msg = new PopUpMessageBox( aParent, tr( "Question" ), aText, YesNoButton );
	msg->setIcon( ":/popup_question" );
	msg->popUp();
	return msg;
}
