#include "PasswordDialog.h"
#include <QLineEdit>
#include <QPushButton>
#include <QVBoxLayout>
#include <QByteArray>
#include <QCryptographicHash>
#include <QtDebug>
#include <QLabel>

PasswordDialog::PasswordDialog( QWidget *aParent, const QString &aPassword, const QString &aText, const QString &aTitle ) :
		QDialog( aParent )
{
	setWindowTitle( aTitle.isNull() ? tr( "Enter password" ) : aTitle );
	setModal( true );

	// Store the password hash to iPasswordHash
	// Note: aPassword is already encoded (if not, that also not our problem here)
	iPasswordHash = aPassword;

	/* Create the layout:

	.--------title---------.
	| text                 |
	|                      |
	| [***_______________] |
	| [Cancel]      [ OK ] |
	'----------------------'
	*/

	QVBoxLayout *layout = new QVBoxLayout;

	if ( !aText.isNull() )
	{
		QLabel *text = new QLabel( aText );
		layout->addWidget( text );
		layout->addStretch();
	}

	iPasswordEdit = new QLineEdit;
	iPasswordEdit->setEchoMode( QLineEdit::Password );
	layout->addWidget( iPasswordEdit );

	QHBoxLayout *buttonLayout = new QHBoxLayout;
	QPushButton *buttonOK = new QPushButton( tr( "OK" ) );
	QPushButton *buttonCancel = new QPushButton( tr( "Cancel" ) );

	buttonLayout->addWidget( buttonCancel );
	buttonLayout->addStretch();
	buttonLayout->addWidget( buttonOK );
	layout->addLayout( buttonLayout );

	// Connect the buttons pressed signals to corresponding slots
	connect( buttonOK, SIGNAL( pressed() ), this, SLOT( okButtonPressed() ) );
	connect( buttonCancel, SIGNAL( pressed() ), this, SLOT( cancelButtonPressed() ) );

	// Enable the layout
	setLayout( layout );
}

PasswordDialog::~PasswordDialog()
{
}

void PasswordDialog::okButtonPressed()
{
	qDebug() << "PasswordDialog::okButtonPressed()";

	// Get md5 hash from the password entered to the dialog
	QString typedPassword = QString( QCryptographicHash::hash( iPasswordEdit->text().toUtf8(), QCryptographicHash::Md5 ).toHex() );

	// Compare the password hashes and emit corresponding signal telling if the password was correct
	if ( iPasswordHash == typedPassword )
	{
		emit passwordEntered( PasswordDialog::Correct );
		qDebug() << "PasswordDialog::okButtonPressed()/tPassword OK";
	}
	else
	{
		emit passwordEntered( PasswordDialog::Incorrect );
		qDebug() << "PasswordDialog::okButtonPressed()/tIncorrect password!";
	}

	// Close the dialog
	close();
}

void PasswordDialog::cancelButtonPressed()
{
	qDebug() << "PasswordDialog::cancelButtonPressed()";

	emit passwordEntered( PasswordDialog::Canceled );

	close();
}

PasswordDialog * PasswordDialog::query( QWidget *aParent, const QString &aPassword, const QString &aText, const QString &aTitle )
{
	// Create a PasswordDialog instance and show it
	PasswordDialog* dlg = new PasswordDialog( aParent, aPassword, aText, aTitle );
	dlg->show();
	return dlg;
}
