#ifndef SCHEDULEWIDGET_H_
#define SCHEDULEWIDGET_H_

#include "ObservedWidget.h"
#include <QDateTime>
#include <QList>
#include "DisplaySettings.h"
#include <QTableWidget>

class QTableWidgetItem;
class Meeting;

//! UserInterface class. Customizes the QTableWidget class.
/*!
 * UserInterface class. inherites QTableWidget to draw a custom table.
 */
class ScheduleTableWidget : public QTableWidget
{
	Q_OBJECT

	//! Container for a meeting.
	struct MeetingContainer
	{
		Meeting* meeting;
		QRect rect;
		bool rectComputed;
	};

public:

	//! Constructor.
	/*!
	 * Constructor to initialize ScheduleTableWidget instance.
	 * \param aRows number of rows
	 * \param aColumns number of columns
	 * \param aParent parent widget
	 */
	ScheduleTableWidget( int aRows, int aColumns, QWidget *aParent = 0 );

	//! Destructor.
	~ScheduleTableWidget();

protected:
	//! Handles drawing of main table area.
	/*!
	 * Handles drawing of main table area.
	 * \param aEvent event
	 */
	void paintEvent( QPaintEvent* aEvent );

	//! Forwards relevant information to activateMeeting().
	/*!
	 * Forwards relevant information to activateMeeting().
	 * \param aEvent event.
	 */
	void tabletEvent( QTabletEvent* aEvent );

	//! Mouse move event.
	/*!
	 * Implemented as empty method for preventing unwanted QTableWidget behavior.
	 * \param aEvent event.
	 */
	void mouseMoveEvent( QMouseEvent* aEvent );

	//! Forwards relevant information to activateMeeting().
	/*!
	 * Forwards relevant information to activateMeeting().
	 * \param aEvent event.
	 */
	void mousePressEvent( QMouseEvent* aEvent );

private:
	//! Populates meeting list.
	/*!
	 * Populates meeting list.
	 */
	void populateMeetingList();

	//! Finds overlapping meetings.
	/*!
	 * Finds overlapping meetings.
	 * \param aDay day.
	 * \param aMeeting meeting to compare.
	 * \param aResult generated list of overlapping meetings, empty if none.
	 * \return true if overlapping meetings were found, otherwise false.
	 */
	bool findOverlappingMeetings( int aDay, Meeting* aMeeting, QList<int>& aResult );

	//! Activates a meeting.
	/*!
	 * Activates a meeting.
	 * \param x x coordinate
	 * \param y y coordinate
	 */
	void activateMeeting( int x, int y );

	//! Computes y coordinate in viewport for a given time.
	/*!
	 * Computes y coordinate in viewport for a given time.
	 * \param aTime time
	 * \return y coordinate
	 */
	int computeViewportY( QTime aTime );

private:

	//! Array of list of meetings, for each day.
	QList<MeetingContainer>* iMeetingsByDay;
};

//! UserInterface class. Defines a custom weekly calendar on the screen. Used by thr WeeklyViewWidget.
/*!
 * UserInterface class. Defines a custom weekly calendar on the screen. Used by thr WeeklyViewWidget.
 */
class ScheduleWidget : public ObservedWidget
{
	Q_OBJECT

	friend class ScheduleTableWidget;

public:

	//! Constructor.
	/*!
	 * Constructor to initialize ScheduleWidget instance.
	 * \param aCurrentDateTime Current date and time.
	 * \param aSettings Display settings.
	 * \param aParent Parent widget.
	 */
	ScheduleWidget( QDateTime aCurrentDateTime, DisplaySettings *aSettings, QWidget *aParent = 0 );

	//! Destructor.
	virtual ~ScheduleWidget();

	//! Gets number of current week.
	/*!
	 * Gets number of current week.
	 * \return Number of week.
	 */
	int	currentWeek();

	//! Gets number of shown week.
	/*!
	 * Gets number of shown week.
	 * \return The week which is currently shown.
	 */
	int	shownWeek();

	//! Gets the first day of shown week.
	/*!
	 * Gets the first day of shown week.
	 * \return First day of the shown week.
	 */
	QDate beginningOfShownWeek();

	//! Gets current meeting.
	/*!
	 * Gets current meeting.
	 * \return current meeting, 0 if none.
	 */
	Meeting* currentMeeting();

	//! Gets meeting at a specified date and time.
	/*!
	 * Gets meeting at a specified date and time.
	 * \param aAt Time when the meeting is.
	 * \return meeting At the given time, 0 if none.
	 */
	Meeting* meeting( QDateTime aAt );

signals:

	//! Signal. Emitted if a meeting is activated.
	/*!
	 * Signal. Emitted if a meeting is activated, i.e. the user clicks on a meeting rectangle.
	 * \param aMeeting Actived meeting.
	 */
	void meetingActivated( Meeting *aMeeting );

	//! Signal. Emitted when the current meeting changes to another.
	/*!
	 * Signal. Emitted when the current meeting changes to another.
	 * \param aNewMeeting New meeting.
	 */
	void currentMeetingChanged( Meeting *aNewMeeting );

	//! Signal. Emitted if the shown week has been changed.
	/*!
	 * Signal. Emitted if the shown week has been changed.
	 * \param aWeekNumber Week number.
	 */
	void shownWeekChanged( int aWeekNumber );

public slots:

	//! Slot. Clears all meetings.
	/*!
	 * Slot. Clears all meetings.
	 */
	void clear();

	//! Slot. Clears meetings based on a range.
	/*!
	 * Slot. Clears meetings based on a range.
	 * \param aFrom Date and time from which the meetings are cleared.
	 * \param aUntil Date and time until which the meetings are cleared.
	 */
	void clear( QDateTime aFrom, QDateTime aUntil );

	//! Slot. Refreshes display.
	/*!
	 * Slot. Refreshes display.
	 */
	void refresh();

	//! Slot. Sets current date and time.
	/*!
	 * Slot. Sets current date and time.
	 * \param aCurrentDateTime Current date and time.
	 */
	void setCurrentDateTime( QDateTime aCurrentDateTime );

	//! Slot. Inserts a meeting to the schedule.
	/*!
	 * Slot. Inserts a meeting to the schedule.
	 * \param aMeeting Meeting to be inserted.
	 */
	void insertMeeting( Meeting *aMeeting );

	//! Slot. Removes a meeting from the schedule.
	/*!
	 * Slot. Removes a meeting from the schedule.
	 * \param aMeeting Meeting to be removed.
	 */
	void removeMeeting( Meeting *aMeeting );

	//! Slot. Updates a meeting in the schedule. (TODO)
	/*!
	 * Slot updates a meeting in the schedule.
	 * \param aMeeting Meeting was updated.
	 */
	void updateMeeting( Meeting *aMeeting ) {};

	//! Slot. Shows previous week.
	/*!
	 * Slot. Shows previous week.
	 */
	void showPreviousWeek();

	//! Slot. Shows current week.
	/*!
	 * Slot. Shows current week.
	 */
	void showCurrentWeek();

	//! Slot. Shows next week.
	/*!
	 * Slot. Shows next week.
	 */
	void showNextWeek();

private:

	//! Computes header row.
	/*!
	 * Computes header row number in schedule table based on given time.
	 * \param aTime Time.
	 * \return header Row.
	 */
	int computeHeaderRow( QTime aTime );

	//! Gets week length as days.
	/*!
	 * Gets week length as days.
	 * \return Lenght of the week in days.
	 */
	int weekLengthAsDays();

	//! Computes proper cell sizes for the schedule table.
	/*!
	 * Computes proper cell sizes for the schedule table.
	 * \param aEvent Resize event.
	 */
	void resizeEvent( QResizeEvent *aEvent );

private:
	//! Schedule table widget.
	ScheduleTableWidget *iScheduleTable;

	//! Current date and time.
	QDateTime iCurrentDateTime;

	//! Meetings.
	QList<Meeting*> iMeetings;

	//! Currently shown week.
	QDate iShownDate;

	//! Starting hour of the schedule.
	int iStartHour;

	//! Number of hours in the schedule.
	int iNumberOfHours;

	//! Variable indicates the length of the week.
	DisplaySettings::DaysInSchedule iDaysInSchedule;

	//! When refresh() was called previously
	QTime iLastRefresh;

	//! Color for a free cell.
	static const QColor sFreeBackground;

	//! Color for a busy cell.
	static const QColor sBusyBackground;

	//! Color for headers.
	static const QColor sHeaderBackground;

	//! Color for current day highlight.
	static const QColor sDayHighlightColor;

	//! Color for current time highlight.
	static const QColor sTimeHighlightColor;

	//! Color for main grid.
	static const QColor sMainGridColor;

	//! Color for half grid.
	static const QColor sHalfGridColor;

	//! Color for frame.
	static const QColor sFrameColor;

	//! Refresh interval.
	static const int sRefreshIntervalInSeconds = 60;
};

#endif /*SCHEDULEWIDGET_H_*/
