#include "ScheduleWidget.h"

#include <QTableWidget>
#include <QHeaderView>
#include <QVBoxLayout>
#include <QTime>
#include <QtDebug>
#include <QResizeEvent>
#include <QPainter>
#include "Meeting.h"

const QColor ScheduleWidget::sFreeBackground = QColor( 192, 238, 189 );
const QColor ScheduleWidget::sBusyBackground = QColor( 238, 147, 17 );
const QColor ScheduleWidget::sHeaderBackground = QColor( Qt::white );
const QColor ScheduleWidget::sDayHighlightColor = QColor( 255, 235, 160 );
const QColor ScheduleWidget::sTimeHighlightColor = QColor( Qt::blue );
const QColor ScheduleWidget::sMainGridColor = QColor( 140, 140, 140 );
const QColor ScheduleWidget::sHalfGridColor = QColor( 195, 195, 195 );
const QColor ScheduleWidget::sFrameColor = QColor( Qt::black );

ScheduleTableWidget::ScheduleTableWidget( int aRows, int aColumns, QWidget *aParent ) :
		QTableWidget( aRows, aColumns, aParent )
{
	ScheduleWidget* schedule = static_cast<ScheduleWidget*>( parent() );

	iMeetingsByDay = new QList<MeetingContainer>[schedule->weekLengthAsDays()];

	setFocusPolicy( Qt::NoFocus );
	setFrameStyle( QFrame::NoFrame );
}

ScheduleTableWidget::~ScheduleTableWidget()
{
	delete[] iMeetingsByDay;
}

void ScheduleTableWidget::paintEvent( QPaintEvent* aEvent )
{
	QTableWidget::paintEvent( aEvent );

	ScheduleWidget* schedule = static_cast<ScheduleWidget*>( parent() );
	QPainter painter( viewport() );
	int rowHeight = rowViewportPosition( 2 ) - rowViewportPosition( 1 ) - 1;
	int columnWidth = columnViewportPosition( 2 ) - columnViewportPosition( 1 ) - 1;

	// draw frame around the table
	QRect viewportRect = viewport()->rect();
	viewportRect.adjust( 0, 0, -1, -1 );
	painter.setPen( ScheduleWidget::sFrameColor );
	painter.drawRect( viewportRect );

	// draw horizontal half grid
	for ( int i = 1; i < rowCount(); ++i )
	{
		int x = columnViewportPosition( 1 );
		int y = rowViewportPosition( i ) + ( rowHeight / 2 ) - 1;
		painter.fillRect( QRect( x, y, width() - x - 1, 1 ), ScheduleWidget::sHalfGridColor );
	}

	// draw horizontal main grid
	for ( int i = 1; i < rowCount(); ++i )
	{
		int y = rowViewportPosition( i ) - 1;
		painter.fillRect( QRect( 1, y, width() - 2, 1 ), ScheduleWidget::sMainGridColor );
	}

	// draw vertical main grid
	for ( int i = 1; i < columnCount(); ++i )
	{
		int x = columnViewportPosition( i ) - 1;
		painter.fillRect( QRect( x, 1, 1, height() - 2 ), ScheduleWidget::sMainGridColor );
	}

	// draw current day highlight
	QPen pen( ScheduleWidget::sDayHighlightColor );
	pen.setWidth( 3 );
	painter.setPen( pen );
	painter.setBrush( Qt::NoBrush );

	for ( int i = 1; i < columnCount(); ++i )
	{
		if ( schedule->iCurrentDateTime.date() == schedule->iShownDate.addDays( i - 1 ) )
		{
			int x = columnViewportPosition( i ) + 1;
			int y = 2;
			int w = columnWidth - 3;
			int h = height() - 5;
			painter.drawRect( x, y, w, h );
			break;
		}
	}

	// draw meetings
	QBrush brush( ScheduleWidget::sBusyBackground );
	painter.setBrush( brush );
	painter.setRenderHint( QPainter::Antialiasing );
	painter.setPen( ScheduleWidget::sFrameColor );
	populateMeetingList();

	for ( int day = 0; day < schedule->weekLengthAsDays(); ++day )
	{
		for ( int i = 0; i < iMeetingsByDay[day].size(); ++i )
		{
			painter.drawRoundRect( iMeetingsByDay[day][i].rect, 20, 20 );
		}
	}

	// draw current time highlight
	painter.setBrush( Qt::NoBrush );
	painter.setRenderHint( QPainter::Antialiasing, false );

	for ( int i = 1; i < columnCount(); ++i )
	{
		if ( schedule->iCurrentDateTime.date() == schedule->iShownDate.addDays( i - 1 ) )
		{
			int x = columnViewportPosition( i ) - 1;
			int y = computeViewportY( schedule->iCurrentDateTime.time() );
			int w = columnWidth + 2;
			int h = 4;
			painter.fillRect( x, y, w, h, ScheduleWidget::sTimeHighlightColor );
			break;
		}
	}
}

void ScheduleTableWidget::tabletEvent( QTabletEvent* aEvent )
{
	activateMeeting( aEvent->x(), aEvent->y() );
}

void ScheduleTableWidget::mouseMoveEvent( QMouseEvent* /* aEvent */ )
{
	// this is overridden as empty method because otherwise
	// unwanted behaviour would occur due to QTableWidget
}

void ScheduleTableWidget::mousePressEvent( QMouseEvent* aEvent )
{
	activateMeeting( aEvent->x(), aEvent->y() );
}

void ScheduleTableWidget::populateMeetingList()
{
	ScheduleWidget* schedule = static_cast<ScheduleWidget*>( parent() );

	for ( int i = 0; i < schedule->weekLengthAsDays(); ++i )
		iMeetingsByDay[i].clear();

	// insert suitable meetings to list
	for ( int i = 0; i < schedule->iMeetings.count(); ++i )
	{
		Meeting* meeting = schedule->iMeetings[i];
		int day = meeting->startsAt().date().dayOfWeek() - 1;
		if (( meeting->startsAt().date().weekNumber() == schedule->iShownDate.weekNumber() ) &&
			  ( day < schedule->weekLengthAsDays() ) &&
			  ( meeting->endsAt().time() > QTime( schedule->iStartHour, 0 ) ) &&
			  ( meeting->startsAt().time() <= QTime( schedule->iStartHour + schedule->iNumberOfHours - 1,  59 ) ) )
		{
			MeetingContainer container;
			container.meeting = meeting;
			container.rect = QRect( 0, 0, 0, 0 );
			container.rectComputed = false;
			iMeetingsByDay[day].append( container );
		}
	}

	// compute meeting rectangles
	for ( int day = 0; day < schedule->weekLengthAsDays(); ++day )
	{
		for ( int i = 0; i < iMeetingsByDay[day].size(); ++i )
		{
			if ( iMeetingsByDay[day][i].rectComputed )
				continue;

			QList<int> meetingIndices;
			findOverlappingMeetings( day, iMeetingsByDay[day][i].meeting, meetingIndices );
			meetingIndices.append( i );

			for ( int j = 0; j < meetingIndices.size(); ++j )
			{
				if ( iMeetingsByDay[day][meetingIndices[j]].rectComputed )
					continue;

				int columnWidth = columnViewportPosition( 2 ) - columnViewportPosition( 1 ) - 1;

				Meeting* meeting = iMeetingsByDay[day][meetingIndices[j]].meeting;
				int x = columnViewportPosition( day + 1 ) + ( int )(( columnWidth / ( float )meetingIndices.size() ) * j );
				int y = computeViewportY( meeting->startsAt().time() );
				int width = ( int )( columnWidth / ( float )meetingIndices.size() + 0.5f );
				int height = computeViewportY( meeting->endsAt().time() ) - y;

				iMeetingsByDay[day][meetingIndices[j]].rect = QRect( x, y, width, height );
				iMeetingsByDay[day][meetingIndices[j]].rectComputed = true;
			}
		}
	}
}

bool ScheduleTableWidget::findOverlappingMeetings( int aDay, Meeting* aMeeting, QList<int>& aResult )
{
	QSet<int> overlapSet;

	// first find meetings that overlap with aMeeting
	for ( int i = 0; i < iMeetingsByDay[aDay].size(); ++i )
	{
		Meeting* other = iMeetingsByDay[aDay][i].meeting;
		if ( aMeeting != other && aMeeting->overlaps( other ) )
			overlapSet.insert( i );
	}

	// then compare overlappiong ones against every meeting to make sure that
	// the returned set can be used to compute rectangles for all cases at once
	foreach( int index, overlapSet )
	{
		Meeting* meetingInSet = iMeetingsByDay[aDay][index].meeting;
		for ( int i = 0; i < iMeetingsByDay[aDay].size(); ++i )
		{
			Meeting* other = iMeetingsByDay[aDay][i].meeting;
			if ( meetingInSet != other && aMeeting != other && meetingInSet->overlaps( other ) )
				overlapSet.insert( i );
		}
	}

	aResult.clear();
	foreach( int index, overlapSet )
	{
		aResult.append( index );
	}

	return !aResult.empty();
}

void ScheduleTableWidget::activateMeeting( int x, int y )
{
	ScheduleWidget* schedule = static_cast<ScheduleWidget*>( parent() );

	for ( int day = 0; day < schedule->weekLengthAsDays(); ++day )
	{
		for ( int i = 0; i < iMeetingsByDay[day].size(); ++i )
		{
			if ( iMeetingsByDay[day][i].rect.contains( x, y ) )
			{
				qDebug() << "Activated meeting at x" << x << "y" << y << ":" << iMeetingsByDay[day][i].meeting->toString();
				emit schedule->meetingActivated( iMeetingsByDay[day][i].meeting );
			}
		}
	}
}

int ScheduleTableWidget::computeViewportY( QTime aTime )
{
	ScheduleWidget* schedule = static_cast<ScheduleWidget*>( parent() );
	int secondsInDisplayDay = schedule->iNumberOfHours * 60 * 60;
	int mainY = rowViewportPosition( 1 ) + 1;
	int mainHeight = height() - mainY - 1;

	return mainY + ( int )(( QTime( schedule->iStartHour, 0 ).secsTo( aTime ) / ( float )secondsInDisplayDay ) * mainHeight );
}

ScheduleWidget::ScheduleWidget( QDateTime aCurrentDateTime, DisplaySettings *aSettings, QWidget *aParent ) :
		ObservedWidget( aParent ),
		iCurrentDateTime( aCurrentDateTime ),
		iStartHour( aSettings->dayStartsAt().hour() ),
		iNumberOfHours( aSettings->dayEndsAt().hour() - aSettings->dayStartsAt().hour() + 1 ),
		iDaysInSchedule( aSettings->daysInSchedule() ),
		iLastRefresh( aCurrentDateTime.time() )
{
	iStartHour = qBound( 0, iStartHour, 23 );
	iNumberOfHours = qBound( 1, iNumberOfHours, 24 - iStartHour );

	iScheduleTable = new ScheduleTableWidget(( iNumberOfHours + 1 ) * 1, weekLengthAsDays() + 1, this );
	iScheduleTable->horizontalHeader()->hide();
	iScheduleTable->verticalHeader()->hide();
	iScheduleTable->setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
	iScheduleTable->setVerticalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
	iScheduleTable->setShowGrid( false );

	QFont font;
	font.setPointSize( 10 );

	// add empty item to top-left corner
	QTableWidgetItem *emptyItem = new QTableWidgetItem();
	emptyItem->setFlags( Qt::ItemIsEnabled );
	emptyItem->setBackgroundColor( sHeaderBackground );
	iScheduleTable->setItem( 0, 0, emptyItem );

	// add empty item to main cell
	QTableWidgetItem *mainItem = new QTableWidgetItem();
	mainItem->setFlags( Qt::ItemIsEnabled );
	mainItem->setBackgroundColor( sFreeBackground );
	iScheduleTable->setItem( 1, 1, mainItem );

	// set row header items
	QTime time( iStartHour, 0 );
	for ( int i = 1; i < iScheduleTable->rowCount(); ++i )
	{
		QTableWidgetItem *item = new QTableWidgetItem( time.toString( "HH:mm" ) );
		item->setTextAlignment( Qt::AlignHCenter );
		item->setFlags( Qt::ItemIsEnabled );
		item->setBackgroundColor( sHeaderBackground );
		item->setFont( font );
		iScheduleTable->setItem( i, 0, item );
		time = time.addSecs( 60 * 60 );
	}

	// set empty column header items, these will be updated in refresh()
	for ( int i = 1; i < iScheduleTable->columnCount(); ++i )
	{
		QTableWidgetItem *item = new QTableWidgetItem();
		item->setTextAlignment( Qt::AlignCenter );
		item->setFlags( Qt::ItemIsEnabled );
		item->setBackgroundColor( sHeaderBackground );
		item->setFont( font );
		iScheduleTable->setItem( 0, i, item );
	}

	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget( iScheduleTable );
	layout->setAlignment( Qt::AlignCenter );
	layout->setMargin( 0 );
	setLayout( layout );

	showCurrentWeek();
}

ScheduleWidget::~ScheduleWidget()
{
	clear();
	delete iScheduleTable;
}

int	ScheduleWidget::currentWeek()
{
	return iCurrentDateTime.date().weekNumber();
}

int	ScheduleWidget::shownWeek()
{
	return iShownDate.weekNumber();
}

QDate ScheduleWidget::beginningOfShownWeek()
{
	return iShownDate.addDays( -1 * iShownDate.dayOfWeek() + 1 );
}

Meeting* ScheduleWidget::currentMeeting()
{
	return meeting( iCurrentDateTime );
}

Meeting* ScheduleWidget::meeting( QDateTime aAt )
{
	for ( int i = 0; i < iMeetings.count(); ++i )
	{
		if ( iMeetings[i]->startsAt() <= aAt && iMeetings[i]->endsAt() >= aAt )
		{
			return iMeetings[i];
		}
	}

	return 0;
}

void ScheduleWidget::clear()
{
	qDebug() << "ScheduleWidget::clear";
	int i = 0;
	while ( !iMeetings.isEmpty() )
	{
		i++;
		// TODO: Should the meetings be deleted before remove?
		// delete iMeetings.takeFirst();
		iMeetings.removeFirst();
	}
	qDebug() << "Deleted " << i << " items";
}

void ScheduleWidget::clear( QDateTime aFrom, QDateTime aUntil )
{
	for ( int i = 0; i < iMeetings.count(); ++i )
	{
		if (( iMeetings[i]->startsAt() >= aFrom && iMeetings[i]->startsAt() <= aUntil ) ||
			  ( iMeetings[i]->startsAt() <= aFrom && iMeetings[i]->endsAt() >= aFrom ) )
		{
			// TODO: Should the meetings be deleted before remove?
			delete iMeetings[i];
			iMeetings.removeAt( i );
			--i;
		}
	}
}

void ScheduleWidget::refresh()
{
	for ( int i = 1; i < iScheduleTable->columnCount(); ++i )
	{
		QTableWidgetItem* item = iScheduleTable->item( 0, i );
		QFont font = item->font();
		item->setText( iShownDate.addDays( i - 1 ).toString( tr( "ddd d MMM" ) ) );

		if ( iCurrentDateTime.date() == iShownDate.addDays( i - 1 ) )
		{
			// mark current day
			item->setBackgroundColor( sDayHighlightColor );
			font.setItalic( true );
			item->setFont( font );
		}
		else
		{
			item->setBackgroundColor( sHeaderBackground );
			font.setItalic( false );
			item->setFont( font );
		}
	}

	// force repaint of the main area
	iScheduleTable->setSpan( 1, 1, iNumberOfHours, weekLengthAsDays() );

	iLastRefresh = iCurrentDateTime.time();
}

void ScheduleWidget::setCurrentDateTime( QDateTime aCurrentDateTime )
{
	Meeting* previous = meeting( iCurrentDateTime );
	Meeting* current = meeting( aCurrentDateTime );
	iCurrentDateTime = aCurrentDateTime;

	if ( iLastRefresh.secsTo( iCurrentDateTime.time() ) > sRefreshIntervalInSeconds )
	{
		// enough time has elapsed since last refresh
		refresh();
	}

	if ( previous != current )
	{
		emit currentMeetingChanged( current );
	}
}

void ScheduleWidget::insertMeeting( Meeting *aMeeting )
{
	Meeting* previous = meeting( iCurrentDateTime );
	iMeetings.append( aMeeting );
	Meeting* current = meeting( iCurrentDateTime );

	qDebug() << "Inserted" << aMeeting->toString();

	refresh();

	if ( previous != current )
	{
		emit currentMeetingChanged( current );
	}
}

void ScheduleWidget::removeMeeting( Meeting *aMeeting )
{
	Meeting* previous = meeting( iCurrentDateTime );

	qDebug() << "Delete" << aMeeting->toString();
	for ( int i = 0; i < iMeetings.count(); ++i )
	{
		if ( iMeetings[i]->equals( aMeeting ) )
		{
			delete iMeetings[i];
			iMeetings.removeAt( i );

			refresh();

			Meeting* current = meeting( iCurrentDateTime );
			if ( previous != current )
				emit currentMeetingChanged( current );

			return;
		}
	}
}

//void ScheduleWidget::updateMeeting( Meeting *aMeeting )
//{
//
//}

void ScheduleWidget::showPreviousWeek()
{
	iShownDate = iShownDate.addDays( -7 );
	refresh();
	emit shownWeekChanged( iShownDate.weekNumber() );
}

void ScheduleWidget::showCurrentWeek()
{
	iShownDate = iCurrentDateTime.date();

	// set weekday to monday
	iShownDate = iShownDate.addDays( -( iShownDate.dayOfWeek() - 1 ) );

	refresh();
	emit shownWeekChanged( iShownDate.weekNumber() );
}

void ScheduleWidget::showNextWeek()
{
	iShownDate = iShownDate.addDays( 7 );
	refresh();
	emit shownWeekChanged( iShownDate.weekNumber() );
}

int ScheduleWidget::computeHeaderRow( QTime aTime )
{
	// map given time to correct header row in the schedule table
	return aTime.hour() - ( iStartHour - 1 );
}

int ScheduleWidget::weekLengthAsDays()
{
	return ( iDaysInSchedule == DisplaySettings::WholeWeekInSchedule ) ? 7 : 5;
}

void ScheduleWidget::resizeEvent( QResizeEvent* /* aEvent */ )
{
	QRect rect = iScheduleTable->contentsRect();
	int rowHeight = ( int )( rect.height() / ( float )iScheduleTable->rowCount() );
	int headerRowHeight = rowHeight;
	int columnWidth = ( int )( rect.width() / ( iScheduleTable->columnCount() - 0.5f ) );
	int headerColumnWidth = columnWidth / 2;

	iScheduleTable->setRowHeight( 0, headerRowHeight );
	for ( int i = 1; i < iScheduleTable->rowCount(); ++i )
	{
		iScheduleTable->setRowHeight( i, rowHeight );
	}

	iScheduleTable->setColumnWidth( 0, headerColumnWidth );
	for ( int i = 1; i < iScheduleTable->columnCount(); ++i )
	{
		iScheduleTable->setColumnWidth( i, columnWidth );
	}

	// resize table so that frame size matches exactly
	int leftMargin = 0, topMargin = 0, rightMargin = 0, bottomMargin = 0;
	iScheduleTable->getContentsMargins( &leftMargin, &topMargin, &rightMargin, &bottomMargin );
	iScheduleTable->resize( columnWidth * iScheduleTable->columnCount() - headerColumnWidth + leftMargin + rightMargin,
					rowHeight * iScheduleTable->rowCount() + topMargin + bottomMargin );
}
