#ifndef MEETINGROOMCOMBO_H_
#define MEETINGROOMCOMBO_H_

#include "ObservedWidget.h"
#include <QWidget>
#include <QList>

class QComboBox;
class Room;

//! Userinterface class. Displays a list of selectable meeting rooms.
/*!
 * Userinterface class. Displays a list of selectable meeting rooms. Customized QComboBox which hides
 * all the not needed functionality of the "base" class.
 */
class MeetingRoomCombo : public ObservedWidget
{
	Q_OBJECT

public:
	//! Constructor.
	/*!
	 * Constructs the MeetingRoomCombo.
	 * \param aRooms List of available rooms.
	 * \param aParent parent of the widget
	 */
	MeetingRoomCombo( QList<Room*> aRooms, QWidget *aParent = 0 );
	//! Destructor.
	virtual ~MeetingRoomCombo();

	//! Number of rooms.
	/*!
	 * \return Number of rooms.
	 */
	int count();
	//! Current index
	/*!
	 * \return Index of currently selected item.
	 */
	int currentIndex();
	//! Current room
	/*!
	 * \return Pointer to the room currently selected.
	 */
	Room* currentRoom();

public slots:
	//! Sets the current room
	/*!
	 *  Sets the current room to match aIndex.
	 * \param aIndex index of item to be selected.
	 */
	void setCurrentIndex( int aIndex );
	//! Sets the current room
	/*!
	 *  Sets the current room to be aRoom.
	 * \param aRoom Pointer to the room to be selected.
	 */
	void setCurrentRoom( Room *aRoom );
	//! Sets the current room
	/*!
	 *  Sets the current room based on room's name.
	 * \param aName Name of the room to be selected.
	 */
	void setCurrentRoomBy( const QString &aName );

	//! Searches for the room.
	/*!
	 *  Searches for the room "aRoom".
	 * \param aRoom Pointer to the room to be searched.
	 * \return Index of room if found. Otherwise negative.
	 */
	int findRoom( Room *aRoom );
	//! Searches for the room.
	/*!
	 *  Searches for the room by its name.
	 * \param aName Name of the room to be searched.
	 * \return Index of room if found. Otherwise negative.
	 */
	int findRoomBy( const QString &aName );

signals:
	//! Signals change of the current room.
	/*!
	 *  Signal is emided when room is changed.
	 * \param aNewIndex Index of the item now selected.
	 */
	void currentIndexChanged( int aNewIndex );
	//! Signals change of the current room.
	/*!
	 *  Signal is emided when room is changed.
	 * \param aNewRoom Pointer to the room now selected.
	 */
	void currentRoomChanged( Room *aNewRoom );

private:
	//! Combobox containing the names of the rooms.
	QComboBox *iRoomCombo;
	//! List of rooms.
	QList<Room*> iRooms;
};

#endif /*MEETINGROOMCOMBO_H_*/
