#ifndef DEVICEMANAGER_H_
#define DEVICEMANAGER_H_

#include <QObject>
#include <QStringList>

class AlarmSender;
class HWKeyListener;
class StartupSettings;
class DeviceDataStorage;
class DeviceConfigurator;

//! DeviceControl class. The main class of the DeviceControl.
/*!
 * DeviceControl class. The main class of the DeviceControl. Responsible to communicate between the
 * BusinessLogic and the device. Takes care of the operation mode changes of the application.
 */
class DeviceManager : public QObject
{
	Q_OBJECT

public:
	//! Enumeration of device modes
	/*!
	 * Enumeration of device modes
	 */
	enum OperationMode
	{
		KioskMode, /*!< Device is in kiosk mode. */
		StandAloneModeInProgress, /*!< Device is in stand alone mode. */
		StandAloneMode, /*!< Device is in stand alone mode. */
		EmptyMode /*!< Application cannot read the mode. */
	};

	//! Enumeration of errors
	/*!
	 * Enumeration of errors
	 */
	enum ErrorCode
	{
		FileCreationFailed, /*!< File couldn't be created. */
		OldAlarmsNotRemoved, /*!< Previously sent old alarm events cannot be removed. */
		NewAlarmsNotSent, /*!< New alarms cannot be sent. */
		NewAlarmsNotStored, /*!< Information about new sent alarms cannot be stored. */
		ScreenSettingsNotStored, /*!< Configuration parameters of screen options cannot be stored. */
		ScreenSettingsNotFetched, /*!< Configuration parameters of screen options cannot be fetched. */
		ScreenSettingsNotChanged, /*!< Configuration parameters of screen options cannot be changed. */
		KeySettingsNotFetched, /*!< Configuration parameters of hw key options cannot be fetched. */
		KeySettingsNotStored, /*!< Configuration parameters of hw key options cannot be stored. */
		KeySettingsNotChanged, /*!< Configuration parameters of hw key options cannot be changed. */
		InitScriptNotChanged, /*!< Init script to auto launch the application was not registered/unregistered */
		ModeNotFetched, /*!< Information about the current operation mode cannot be fetched */
		ModeNotStored, /*!< Information about the new opration mode cannot be stored */
		DeviceNotRestarted /*!< Device cannot be restarted */
	};

public:
	//! Constructor.
	/*!
	 * Constructor to initialize a DeviceManager instance.
	 * \param aSettings Pointer to the start up configuration settings.
	 */
	DeviceManager( StartupSettings *aSettings );
	//! Destructor.
	virtual ~DeviceManager();
	//! Gets the current operation mode.
	/*!
	 * Gets the current operation mode of the application.
	 * \return enum value of the current operation mode.
	 */
	DeviceManager::OperationMode currentOperationMode();
	//! Gets the string value of an operation mode.
	/*!
	 * Gets the string value of an operation mode.
	 * \param aMode The enum value of the operation mode.
	 * \return QString value of the current operation mode.
	 */
	QString operationModeToString( OperationMode aMode );

signals:
	//! Signal. Emitted if changing the operation mode fails.
	/*!
	 * Signal. Emitted if changing the operation mode fails.
	 * \param aErrortext The cause of the failure.
	 */
	void changeModeFailed( QString &aErrortext );
	//! Signal. Emitted if user tries to change the operation mode.
	/*!
	 * Signal. Emitted if user tries to change the operation mode.
	 * \param aMode The operation mode that user wants to activate.
	 */
	void changeModeOrdered( DeviceManager::OperationMode aMode );
	//! Signal. Emitted if error occured and error message must be shown on UI.
	/*!
	 * Signal. Emitted if error occured and error message must be shown on UI.
	 * \param aErrorMessage The message.
	 */
	void error( const QString &aErrorMessage );

public slots:
	//! Slot. Changes the operation mode.
	/*!
	 * Slot. Changes the operation mode.
	 */
	void changeMode();

private slots:
	//! Slot. Handles "full screen"-hardware key presses.
	/*!
	 * Slot. Handles "full screen"-hardware key presses. Checks the current operation mode and concludes
	 * the next (desired) operation mode to be set. Emits a changeModeOrdered( DeviceManager::OperationMode )
	 * signal.
	 */
	void HWKeyFullScreenPressed();
	//! Slot. Handles errors.
	/*!
	 * Slot. Handles errors and informs the UI by emitting the error() signal with the message in
	 * parameter.
	 * \param aErrorCode The error code.
	 * \param aAddInfo The possible additional error text.
	 */
	void errorHandler( DeviceManager::ErrorCode aErrorCode, QString aAddInfo = "" );

private:
	//! Updates the internal indicator of the current operation mode.
	/*!
	 * Updates the internal indicator of the current operation mode by asking the DeviceDataStorage to
	 * read it from the internal data storage. Calls finalizeStandAloneMode() method in case the mode is
	 * StandAloneModeInProgress.
	 * \return True if operation mode fetching succeeds; otherwise, false.
	 */
	bool setCurrentOperationMode();
	//! Stores the current operation mode.
	/*!
	 * Stores the current operation mode by asking the DeviceDataStorage to write it to the internal
	 * data storage.
	 * \param aMode The operation mode that user wants to activate.
	 * \return True if operation mode storing succeeds; otherwise, false.
	 */
	bool storeOperationMode( OperationMode aMode );
	//! Asks DeviceConfigurator to remove the deactivate script of the application.
	/*!
	 * Asks DeviceConfigurator to remove the deactivate script of the application. Also asks
	 * DeviceDataStorage to store the current operation mode (StandAloneMode) .
	 * \return True if operation mode storing and deactivation of the init script succeed; otherwise, false.
	 */
	bool finalizeStandAloneMode();
	//! Connects/disconnects the HWKeyListener signals to the private HWKeyFullScreenPressed() slot.
	/*!
	 * Connects/disconnects the HWKeyListener signals to the private HWKeyFullScreenPressed() slot. In case
	 * a signal is caught the connection is disabled until the signal handling is finished.
	 * \param aHandle indicates if the signals should be connected or not.
	 */
	void handleKeyPresses( bool aHandle );

private:
	AlarmSender *iAlarmSender;
	HWKeyListener *iHWKeyListener;
	StartupSettings *iSettings;
	DeviceDataStorage *iDataStorage;
	DeviceConfigurator *iConfigurator;

	OperationMode iMode;
	bool iSendErrorMessages;

};

#endif /*DEVICEMANAGER_H_*/
