#include "DeviceManager.h"
#include "AlarmSender.h"
#include "HWKeyListener.h"
#include "StartupSettings.h"
#include "DeviceDataStorage.h"
#include "DeviceConfigurator.h"

#include <QtDebug>
#include <QStringList>

DeviceManager::DeviceManager( StartupSettings *aSettings )
{
	qDebug() << "DeviceManager::DeviceManager( StartupSettings * )";

	iSettings = aSettings;
	iSendErrorMessages = true;

	iDataStorage = new DeviceDataStorage();
	connect( iDataStorage, SIGNAL( dataStorageInitFailed( DeviceManager::ErrorCode ) ),
		   this, SLOT( errorHandler( DeviceManager::ErrorCode ) ) );

	iAlarmSender = new AlarmSender( iDataStorage );
	connect( iAlarmSender, SIGNAL( alarmSendingFailed( DeviceManager::ErrorCode, QString& ) ),
		   this, SLOT( errorHandler( DeviceManager::ErrorCode, QString& ) ) );

	iConfigurator = new DeviceConfigurator( iDataStorage );
	connect( iConfigurator, SIGNAL( configuringError( DeviceManager::ErrorCode ) ),
		   this, SLOT( errorHandler( DeviceManager::ErrorCode ) ) );

	if ( !iDataStorage->initDataStorage() || !setCurrentOperationMode() )
		iMode = EmptyMode;

	iHWKeyListener = new HWKeyListener();
	handleKeyPresses( true );
}

DeviceManager::~DeviceManager()
{
	qDebug() << "DeviceManager::~DeviceManager()";
}

DeviceManager::OperationMode DeviceManager::currentOperationMode()
{
	return iMode;
}

QString DeviceManager::operationModeToString( OperationMode aMode )
{
	switch ( aMode )
	{
		case KioskMode:
			return "Kiosk-mode";
		case StandAloneMode:
			return "Stand Alone-mode";
		default:
			return "";
	}
}

void DeviceManager::changeMode()
{
	qDebug() << "void DeviceManager::changeMode()";

	// no more key presses before this one is handled
	handleKeyPresses( false );

	switch ( iMode )
	{

		case EmptyMode:
			// error occured. Mode cannot be changed
			errorHandler( ModeNotFetched );
			break;
		case StandAloneMode:

			// change to KioskMode

			// check if auto turn on/off functionality enabled and send turn on/off alarm events to alarm daemon
			if ( iSettings->isPowersavingEnabled() )
			{
				if ( !iAlarmSender->sendAlarms( iSettings->turnOnAt(), iSettings->turnOffAt() ) )
				{
					handleKeyPresses( true );
					return;	//this is critical so returning if no success
				}
			}

			// - disable the certain hw keys (only "home"-hw-key at the moment)
			// - register init script to launch the application when ever the device is launched
			// - disable the screen "auto-switch-off" and "dimming"
			// - store info about the new operation mode
			if ( !iConfigurator->toggleHWKeys( false ) ||
				  !iConfigurator->toggleInitScript( true ) ||
				  !iConfigurator->toggleScreenSwitchOff( false ) ||
				  !this->storeOperationMode( KioskMode ) ||
				  !iConfigurator->restartDevice() )
			{

				// we have to roll back if something fails
				// of course rolling back may fail as well but it is impossible to catch
				iSendErrorMessages = false;
				iAlarmSender->removeAlarms();
				iConfigurator->toggleHWKeys( true );
				iConfigurator->toggleInitScript( false );
				iConfigurator->toggleScreenSwitchOff( true );
				iSendErrorMessages = true;
				handleKeyPresses( true );
				return;
			}
			break;

		case KioskMode:
			// change to StandAloneInProgress mode

			// - enable the certain hw keys (only "home"-hw-key at the moment)
			// - unregister the init script
			// - enable the screen "auto-switch-off" and "dimming"
			// - store info about the new operation mode
			if (	!iConfigurator->toggleHWKeys( true ) ||
				  !iConfigurator->toggleScreenSwitchOff( true ) ||
				  !this->storeOperationMode( StandAloneModeInProgress ) ||
				  !iAlarmSender->removeStoredAlarms() ||
				  !iConfigurator->restartDevice() )
			{
				// we have to roll back if something fails
				// of course rolling back may fail as well but it is impossible to catch
				iSendErrorMessages = false;
				iConfigurator->toggleHWKeys( false );
				iConfigurator->toggleInitScript( true );
				iConfigurator->toggleScreenSwitchOff( false );
				this->storeOperationMode( KioskMode );
				iSendErrorMessages = true;
				handleKeyPresses( true );
				return;
			}

			break;
		default: // StandAloneModeInProgress should never come in question
			break;

	}
}

bool DeviceManager::setCurrentOperationMode()
{
	qDebug() << "DeviceManager::currentOperationMode()";
	QStringList modeResult;
	if ( !iDataStorage->readData( iDataStorage->dataSectionToString( DeviceDataStorage::DeviceMode ), modeResult ) )
	{
		errorHandler( ModeNotFetched );
		return false;
	}
	else
	{
		if ( !modeResult.empty() )
		{
			iMode = ( OperationMode )modeResult.at( 0 ).toInt();
			if ( iMode == StandAloneModeInProgress )
			{
				if ( !finalizeStandAloneMode() )
					iMode = EmptyMode;
				else
					iMode = StandAloneMode;
			}
		}
		else //this must be the first time, so no mode info saved yet
			iMode = StandAloneMode;
	}
	return true;
}

bool DeviceManager::storeOperationMode( OperationMode aMode )
{
	qDebug() << "DeviceManager::storeOperationMode( const OperationMode & )";
	QStringList modeStrList;
	QString str;
	modeStrList.append( str.append( QString( "%1" ).arg( aMode ) ) );
	if ( !iDataStorage->storeData( iDataStorage->dataSectionToString( DeviceDataStorage::DeviceMode ), modeStrList ) )
	{
		errorHandler( ModeNotStored );
		return false;
	}
	return true;
}

bool DeviceManager::finalizeStandAloneMode()
{
	qDebug() << "DeviceManager::finalizeStandAloneMode()";
	if ( !storeOperationMode( StandAloneMode ) || !iConfigurator->toggleInitScript( false ) )
		return false;
	return true;
}

void DeviceManager::handleKeyPresses( bool aHandle )
{
	qDebug() << "DeviceManager::handleKeyPresses( bool )";
	if ( !aHandle )
		disconnect( iHWKeyListener, SIGNAL( HWKeyFullScreenPressed() ), this, SLOT( HWKeyFullScreenPressed() ) );
	else
		connect( iHWKeyListener, SIGNAL( HWKeyFullScreenPressed() ), this, SLOT( HWKeyFullScreenPressed() ) );
}

void DeviceManager::HWKeyFullScreenPressed()
{
	qDebug() << "DeviceManager::HWKeyFullScreenPressed()";
	OperationMode nextMode;
	switch ( iMode )
	{
		case KioskMode:
			nextMode = StandAloneMode;
			break;
		case StandAloneMode:
			nextMode = KioskMode;
			break;
		default:
			nextMode = EmptyMode;
			break;
	}
	if ( nextMode != EmptyMode )
		emit changeModeOrdered( nextMode );
}

void DeviceManager::errorHandler( DeviceManager::ErrorCode aErrorCode, QString aAddInfo )
{
	qDebug() << "DeviceManager::errorHandler( ErrorCode, QString & )";
	qDebug() << "DeviceManager::errorHandler: aErrorCode == " << aErrorCode << " " << aAddInfo;
	if ( !iSendErrorMessages )
		return;

	QString errorMessage = "Failed to change the operation mode.\n";

	switch ( aErrorCode )
	{
		case FileCreationFailed:
			errorMessage.append( "Error creating internal file storage." );
			break;
		case OldAlarmsNotRemoved:
			errorMessage.append( "Error removing old alarm events." );
			break;
		case NewAlarmsNotSent:
			errorMessage.append( "Error sending alarm events." );
			break;
		case NewAlarmsNotStored:
			errorMessage.append( "Error storing data about sent alarm events. Removing the already sent alarm events." );
			break;
		case KeySettingsNotFetched:
			errorMessage.append( "Error fetching data of original hardware key settings." );
			break;
		case KeySettingsNotStored:
			errorMessage.append( "Error storing data of original hardware key settings." );
			break;
		case KeySettingsNotChanged:
			errorMessage.append( "Error changing hardware key settings." );
			break;
		case InitScriptNotChanged:
			errorMessage.append( "Init script to auto launch the application was not registered/unregistered." );
			break;
		case ModeNotFetched:
			errorMessage.append( "The application failed to know the current mode of itself." );
			break;
		case ModeNotStored:
			errorMessage.append( "The application failed to store the current mode of itself." );
			break;
		case ScreenSettingsNotStored:
			errorMessage = "Error storing data of original automatic screen switching-off and dimming ";
			errorMessage.append( "parameter values. Using the default values instead." );
			break;
		case ScreenSettingsNotFetched:
			errorMessage = "Error fetching data of original automatic screen switching-off and dimming ";
			errorMessage.append( "parameter values. Using the default values instead." );
			break;
		case ScreenSettingsNotChanged:
			errorMessage.append( "Error changing automatic screen switching-off and dimming parameter values." );
			break;
		case DeviceNotRestarted:
			errorMessage.append( "Error restarting the device." );
			break;
	}
	if ( aAddInfo != "" )
		errorMessage.append( "\n" + aAddInfo );
	emit error( errorMessage );
}
