#ifndef DEVICEDATASTORAGE_H_
#define DEVICEDATASTORAGE_H_

#include "DeviceManager.h"
#include "DeviceConstants.h"

#include <QObject>

#include <QHash>
#include <QStringList>

//! DeviceControl class. Reads and stores data form/to the internal data storage.
/*!
 * DeviceControl class. Reads and stores data form/to the internal data storage. Datastorage is an
 * ini file which contains information about sent alarms, device screen settings, device hardware key
 * settings and current application operation mode. The class can be used to manipulate any other ini
 * typed files as well.
 */
class DeviceDataStorage : public QObject
{
	Q_OBJECT

public:
	//! Enumeration of data sections
	/*!
	 * Enumeration of data sections
	 */
	enum DataSection
	{
		Alarms, /*!< Alarms section. */
		ScreenSettings, /*!< Screen option section. */
		KeySettings, /*!< HW key option section. */
		DeviceMode /*!< Device mode section. */
	};

public:
	//! Constructor.
	/*!
	 * Constructor to initialize a DeviceDataStorage instance
	 */
	DeviceDataStorage();
	//! Destructor.
	virtual ~DeviceDataStorage();
	//! Creates the internal data storage file.
	/*!
	 * Creates the internal data storage file.
	 * \return True if file creation succeeds; otherwise false.
	 */
	bool initDataStorage();
	//! Stores data to an ini file.
	/*!
	 * Stores data to an ini file. Calls readDataFromFile( ... ) method to receive the current content of
	 * the file and replaces the data under the certain section or creates a completely new section with
	 * the data. After that it calls the writeDataToFile( ... ) to store the modified content.
	 * \param aSection The data section under which the data needs to be stored.
	 * \param aData The data to be stored.
	 * \param aFilename The file to modify.
	 * \return True if file modifying succeeds; otherwise false.
	 */
	bool storeData( QString aSection, const QStringList &aData, const QString &aFilename = DataStoragePath );
	//! Reads data from an ini file.
	/*!
	 * Reads data from an ini file. Calls readDataFromFile( ... ) method to receive the current content
	 * of the file and finds the data under the certain section and stores it to the aData parameter.
	 * \param aSection The data section under which the data needs to be red.
	 * \param aData The parameter to place the found data.
	 * \param aFilename The file to read.
	 * \return True if file reading succeeds; otherwise false.
	 */
	bool readData( QString aSection, QStringList &aData, const QString &aFilename = DataStoragePath );
	//! Transform the data section enum value to QString.
	/*!
	 * Transform the data section enum value to QString. Is used only when handling the internal data
	 * storage file.
	 * \param aSection The data section enum value to be transformed.
	 * \return The QString value that corresponds the enum value.
	 */	
	QString dataSectionToString( DataSection aDataSection );

signals:
	//! Signal. Emitted if error occured during internal data storage file creation.
	/*!
	 * Signal. Emitted if error occured during internal data storage file creation.
	 * \param aErrorCode The error code.
	 */
	void dataStorageInitFailed( DeviceManager::ErrorCode aErrorCode );

private:
	//! Replaces the original ini file with the temporary ini file that conatins the required changes.
	/*!
	 * Replaces the original ini file with the temporary ini file that conatins the required changes.
	 * Uses the static DeviceConfigurator method systemIO( ... ) to execute the rename script
	 * qtmeetings-rename.
	 * \param aFilenameTmp The temporary file.
	 * \param aFilename The actual file.
	 * \return True if file renaming succeeds; otherwise false.
	 */
	bool renameFile( const QString &aFilenameTmp, const QString &aFilename );
	//! Reads data from an ini file.
	/*!
	 * Reads data from an ini file. Does the actual job to read the data from the ini file.
	 * \param aDataHash Place to put the content of the file. Key of the hash is for the section and 
	 * value is for the actual data.
	 * \param aSectionOrder Place to keep track of the sections and their orders in the file. QHash doesn't
	 * keep track of the order of the keys.
	 * \param aFilename The file to read.
	 * \return True if file reading succeeds; otherwise false.
	 */
	bool readDataFromFile( QHash<QString, QStringList> &aDataHash, QStringList &aSectionOrder, const QString &aFilename );
	//! Stores data to an ini file.
	/*!
	 * Stores data to an ini file. Does the actual job to store the data to the ini file.
	 * \param aDataHash To read the data to be stored. Key of the hash is for the section and value is for
	 * the actual data.
	 * \param aSectionOrder Contains the correct oredr of the sections to be stored.
	 * \param aFilename The file to store the data.
	 * \return True if file writing succeeds; otherwise false.
	 */
	bool writeDataToFile( const QHash<QString, QStringList> &aDataHash, const QStringList &aSectionOrder, const QString &aFilename );

};

#endif /*DEVICEDATASTORAGE_H_*/
