#include "DeviceDataStorage.h"
#include "DeviceConfigurator.h"

#include <QtDebug>
#include <QFile>
#include <QHash>
#include <QStringList>
#include <QTextStream>

DeviceDataStorage::DeviceDataStorage()
{
	qDebug() << "DeviceDataStorage::DeviceDataStorage()";
}

bool DeviceDataStorage::initDataStorage()
{
	qDebug() << "DeviceDataStorage::initDataStorage()";
	//create the conf file if not yet created
	QFile file( DataStoragePath );	
	
	if ( !file.exists() ) {
		
		// to avoid possible permission errors, create temp data file and rename it after finished
		QString DataStoragePathTmp = "/tmp/" + ( DataStoragePath.split( '/' ) ).back() + ".tmp";
		QFile fileTmp( DataStoragePathTmp );
		
		if ( !fileTmp.open( QIODevice::WriteOnly | QIODevice::Text ) 
				|| !fileTmp.setPermissions( QFile::ReadUser | QFile::WriteUser | QFile::ReadGroup | QFile::WriteGroup | QFile::ReadOwner | QFile::WriteOwner | QFile::ReadOther | QFile::WriteOther ) ) {
			emit dataStorageInitFailed( DeviceManager::FileCreationFailed );
			return false;
		}
		else {
			fileTmp.close();
			file.close();
			if ( !renameFile( DataStoragePathTmp, DataStoragePath ) )
				return false;
		}
	}
	return true;
}

DeviceDataStorage::~DeviceDataStorage()
{
	qDebug() << "DeviceDataStorage::~DeviceDataStorage()";
}

bool DeviceDataStorage::readData( const QString aSection, QStringList &aData, const QString &aFilename )
{
	qDebug() << "DeviceDataStorage::readData( QString, QStringList &, const QString &)";
	QHash<QString, QStringList> allData;
	QStringList sectionOrder; //not needed in this case

	if ( !readDataFromFile( allData, sectionOrder, aFilename ) )
		return false;

	QHash<QString, QStringList>::iterator i;
	for ( i = allData.begin(); i != allData.end(); ++i )
	{
		if ( i.key() == aSection )
		{
			aData = i.value();
			break;
		}
	}
	return true;
}

bool DeviceDataStorage::storeData( QString aSection, const QStringList &aData, const QString &aFilename )
{
	qDebug() << "DeviceDataStorage::storeData( QString, const QStringList &, const QString & )";
	QHash<QString, QStringList> allData;
	QHash<QString, QStringList> allDataNew;
	QStringList sectionOrder;
	bool sectionFound = false;

	if ( !readDataFromFile( allData, sectionOrder, aFilename ) )
		return false;

	// replace data under the certain section
	QHash<QString, QStringList>::iterator i;
	for ( i = allData.begin(); i != allData.end(); ++i ) {
		if ( i.key() == aSection ) {
			allDataNew.insert( i.key(), aData );
			sectionFound = true;
		}
		else
			allDataNew.insert( i.key(), i.value() );
	}

	// data with this section is not yet in the file
	if ( !sectionFound ) {
		allDataNew.insert( aSection, aData );
		sectionOrder.append( aSection );
	}
	
	// to avoid possible permission errors, write the data to new file
	QString filenameTmp = "/tmp/" + ( aFilename.split( '/' ) ).back() + ".tmp";

	if ( !writeDataToFile( allDataNew, sectionOrder, filenameTmp ) )
		return false;
	if ( !renameFile( filenameTmp, aFilename ) )
		return false;

	return true;
}

bool DeviceDataStorage::renameFile( const QString &aFilenameTmp, const QString &aFilename )
{
	qDebug() << "DeviceDataStorage::renameFile( const QString &, const QString & )";
	
	QByteArray result;
	if( !DeviceConfigurator::whoAmI( result ) )
		return false;
	
	QString command;
	QStringList empty;
	if( result != "root" )
		command = "sudo ";
	
	command.append( BinPath + RenameScript + " " + aFilenameTmp + " " + aFilename );
	
	if( !DeviceConfigurator::systemIO( command, empty, result ) )
		return false;

	return true;
}

bool DeviceDataStorage::readDataFromFile( QHash<QString, QStringList> &aDataHash, QStringList &aSectionOrder, const QString &aFilename )
{
	qDebug() << "DeviceDataStorage::readDataFromFile( QHash<QString,QStringList> &, const QString & )";
	QFile file( aFilename );

	if ( !file.open( QIODevice::ReadOnly | QIODevice::Text ) )
		return false;

	QTextStream in( &file );
	QString dataSection = "";
	QStringList data;
	bool firstSection = true;

	while ( !in.atEnd() )
	{
		QString line = in.readLine();
		if ( line.at( 0 ) == QChar( '[' ) && line.at( line.length() - 1 ) == QChar( ']' ) )
		{
			if ( !firstSection )
				aDataHash.insert( dataSection, data );
			line.replace( QString( "[" ), QString( "" ) );
			line.replace( QString( "]" ), QString( "" ) );
			dataSection = line;
			data.clear();
			firstSection = false;
			aSectionOrder.append( dataSection );
		}
		else
			data.append( line );
	}
	if ( dataSection != "" )	//the file wasn't empty, saving the last section
		aDataHash.insert( dataSection, data );
	file.close();

	return true;
}

bool DeviceDataStorage::writeDataToFile( const QHash<QString, QStringList> &aDataHash, const QStringList &aSectionOrder, const QString &aFilename )
{
	qDebug() << "DeviceDataStorage::writeDataToFile( QHash<QString,QStringList> &, const QString & )";
	
	QFile file( aFilename );

	if ( !file.open( QIODevice::WriteOnly | QIODevice::Text ) )
		return false;

	QTextStream out( &file );

	QHash<QString, QStringList>::const_iterator i;
	for ( int j = 0; j < aSectionOrder.size(); ++j )
	{
		for ( i = aDataHash.begin(); i != aDataHash.end(); ++i )
		{
			if ( aSectionOrder.at( j ) == i.key() )
			{
				out << "[" << i.key() << "]\n";
				QStringList list = i.value();
				for ( int j = 0; j < list.size(); ++j )
					out << list.at( j ) << "\n";
				break;
			}
		}
	}
	file.close();

	return true;
}

QString DeviceDataStorage::dataSectionToString( DataSection aDataSection )
{
	qDebug() << "DeviceDataStorage::dataSectionToString( DataSection )";
	switch ( aDataSection )
	{
		case Alarms:
			return "Alarms";
		case ScreenSettings:
			return "ScreenSettings";
		case KeySettings:
			return "KeySettings";
		default: //DeviceMode
			return "DeviceMode";
	}
}
