#ifndef DEVICECONFIGURATOR_H_
#define DEVICECONFIGURATOR_H_

#include "DeviceDataStorage.h"

//! DeviceControl class. Configures the device behavior.
/*!
 * DeviceControl class. DeviceControl class. Configures the device behavior. Updates the device system 
 * files, activates/deactivates the application initialization script, uses the gconftool to configure
 * some parameters of the device and uses the devstopper binary to restart the device to activate
 * the changes. The changes it makes affect to the hardware keys, the screen dimming enabling/disabling
 * and the automated turning on/off the device. These are all supported features of the "kiosk mode".  
 */
class DeviceConfigurator : public QObject 
{
	Q_OBJECT

public:
	//! Constructor.
	/*!
	 * Constructor to initialize an AlarmSender instance
	 * \param aDataStorage Pointer to the DataStorage class instance.
	 */
	DeviceConfigurator( DeviceDataStorage *aDataStorage );
	//! Destructor.
	virtual ~DeviceConfigurator();
	//! Toggles the screen switching on/off and dimming options of the device.
	/*!
	 * Toggles the screen switching on/off and dimming options of the device. Asks DeviceDataStorage to
	 * read/store the screen related original parameters. Uses systemIO( ... ) method to execute
	 * gconftool system commands to change the configurations.
	 * \param aEnable Indicates if the screen switchin off and dimming are wanted to be enabled.
	 * \return True if configuring of screen options succeeds; otherwise, false. 
	 */
	bool toggleScreenSwitchOff( bool aEnable );
	//! Toggles the enabling of the "home"-hardware key of the device.
	/*!
	 * Toggles the enabling of the "home"-hardware key of the device. Asks DeviceDataStorage to
	 * read/store the hardware key related original parameters. Uses the DeviceDataStorage to manipulate
	 * the device system file /etc/mce/mce.ini to enable/disable the hw key. 
	 * \param aEnable Indicates if the "home"-hardware key is wanted to be enabled.
	 * \return True if configuring of the hw-key succeeds; otherwise, false. 
	 */
	bool toggleHWKeys( bool aEnable );
	//! Toggles the activation of the application initialization script.
	/*!
	 * Toggles the activation of the application initialization script. Uses the systemIO( ... ) method 
	 * to execute update-rc.d system commands to activate/deactivate the init script qtmeetings-launcher. 
	 * \param aEnable Indicates if the script is wanted to be activated/deactivated
	 * \return True if activating/deactivating of the script succeeds; otherwise, false. 
	 */
	bool toggleInitScript( bool aEnable );
	//! Restarts the device.
	/*!
	 * Restarts the device by using the systemIO( ... ) method to execute devstopper executable.
	 * \return True if resrating of the device succeeds; otherwise, false.
	 */
	bool restartDevice();
	//! Resolves the currently used account of the device.
	/*!
	 * Resolves the currently used account of the device by using the systemIO( ... ) method to execute 
	 * whoami system command.
	 * \return True if resolving the account succeeds; otherwise, false.
	 */
	static bool whoAmI( QByteArray &aName );
	//! Consists and executes a system command and stores the received output.
	/*!
	 * Consists and executes a system command and stores the received output.
	 * \param aCommand The whole system command or just the "binary"-part of the command
	 * \param aArgs The possible arguments for the "binary"
	 * \param aResult The place to put the output of the command 
	 * \return True if executing the system command succeeds; otherwise, false.
	 */
	static bool systemIO( const QString &aCommand, const QStringList &aArgs, QByteArray &aResult );

signals:
	//! Signal. Emitted if error occured during configuring the device.
	/*!
	 * Signal. Emitted if error occured during configuring the device.
	 * \param aErrorCode The error code.
	 */
	void configuringError( DeviceManager::ErrorCode aErrorCode );

private:
	DeviceDataStorage *iDataStorage;
};

#endif /*DEVICECONFIGURATOR_H_*/
