#include "DeviceConfigurator.h"
#include "DeviceDataStorage.h"
#include "DeviceConstants.h"

#include <QtDebug>
#include <QProcess>

DeviceConfigurator::DeviceConfigurator( DeviceDataStorage *aDataStorage )
{
	qDebug() << "DeviceConfigurator( DeviceDataStorage * )";
	iDataStorage = aDataStorage;
}

DeviceConfigurator::~DeviceConfigurator()
{
	qDebug() << "~DeviceConfigurator()";
}

bool DeviceConfigurator::toggleScreenSwitchOff( bool aEnable )
{
	qDebug() << "DeviceConfigurator::toggleScreenSwitchOff( bool )";
	QString command = "gconftool-2";
	QStringList args;
	QByteArray result;
	QStringList confs;
	QStringList defParams;
	confs << "/system/osso/dsm/display/display_blank_timeout"
	<< "/system/osso/dsm/display/display_dim_timeout"
	<< "/apps/osso/applet/osso-applet-display/turn_off_display"
	<< "/apps/osso/applet/osso-applet-display/brightness_period";
	defParams << "300" << "120" << "300" << "120";
	QStringList origValues;

	if ( !aEnable )
	{
		//disabling the screen "auto-switch-off" and "dimming"

		//using gconftool-2 to get the current values for related configurations
		for ( int i = 0; i < confs.size(); ++i )
		{
			args.clear();
			args << "-g" << confs[i];
			if ( systemIO( command, args, result ) )
			{
				if ( result.toLong() != 0 )
					origValues.append( result );
			}
		}
		if ( origValues.size() == confs.size() )  //values succesfully fetched, now trying to store them
		{
			if ( !iDataStorage->storeData( iDataStorage->dataSectionToString( DeviceDataStorage::ScreenSettings ), origValues ) )
				emit configuringError( DeviceManager::ScreenSettingsNotStored );
		}
		else   //values not fetched, using the default values instead
		{
			emit configuringError( DeviceManager::ScreenSettingsNotFetched );
			origValues.clear();
			for ( int i = 0; i < defParams.size(); ++i )
				origValues.append( defParams.at( i ) );
		}

		//using gconftool-2 to change the related configurations
		for ( int i = 0; i < confs.size(); ++i )
		{
			args.clear();
			args << "-s" << confs[i] << "--type=int" << "6000000";
			if ( !systemIO( command, args, result ) ) {
				emit configuringError( DeviceManager::ScreenSettingsNotChanged );
				return false;
			}
		}
	}
	else
	{
		//setting the screen's "auto-switch-off" and "dimming" settings back as they were

		//reading stored data from internal config file
		if ( !iDataStorage->readData( iDataStorage->dataSectionToString( DeviceDataStorage::ScreenSettings ), origValues ) )
		{
			//cannot read, using the default values instead
			emit configuringError( DeviceManager::ScreenSettingsNotFetched );
			for ( int i = 0; i < defParams.size(); ++i )
				origValues.append( defParams.at( i ) );
		}
		for ( int i = 0; i < origValues.size(); ++i )
		{
			args.clear();
			args << "-s" << confs[i] << "--type=int" << origValues.at(i);
			if ( !systemIO( command, args, result ) ) {
				emit configuringError( DeviceManager::ScreenSettingsNotChanged );
				return false;
			}
		}
	}
	return true;
}

bool DeviceConfigurator::toggleHWKeys( bool aEnable )
{
	qDebug() << "DeviceConfigurator::toggleHWKeys( bool )";
	QStringList mceLines;
	QStringList mceLinesNew;
	QString mceSection = "HomeKey";
	QStringList params;
	params << "HomeKeyShortAction" << "HomeKeyLongAction";

	// using the DeviceDataStorage exceptionally for reading data from an external conf file
	// /etc/mce/mce.ini
	if ( !iDataStorage->readData( mceSection, mceLines, McePath ) )
	{
		emit configuringError( DeviceManager::KeySettingsNotFetched );
		return false;
	}

	if ( !aEnable )
	{
		// disabling the "home"-hw-key

		QStringList mceLinesToStore;
		for ( int i = 0; i < mceLines.size(); ++i )
		{
			QStringList mceLine = mceLines.at( i ).split( '=' );
			QString param = mceLine.at( 0 ).trimmed();
			//check if this is the correct parameter to store and change
			if ( params.contains( param ) )
			{
				for ( int j = 0; j < params.size(); ++j )
				{
					if ( params.at( j ) == param )
					{
						mceLinesToStore.append( mceLines.at( i ) );
						mceLinesNew.append( param + "=disabled" );
						break;
					}
				}
			}
			else
				mceLinesNew.append( mceLines.at( i ) );
		}

		// storing the mce conf file lines
		if ( !iDataStorage->storeData( iDataStorage->dataSectionToString( DeviceDataStorage::KeySettings ), mceLinesToStore ) )
		{
			emit configuringError( DeviceManager::KeySettingsNotStored );
			return false;
		}
	}
	else
	{
		// setting the "home"-hw-key settings back as they were

		// reading the stored mce conf file lines
		QStringList storedMceLines;
		if ( !iDataStorage->readData( iDataStorage->dataSectionToString( DeviceDataStorage::KeySettings ), storedMceLines ) )
		{
			emit configuringError( DeviceManager::KeySettingsNotFetched );
			return false;
		}

		bool paramFound = false;
		for ( int i = 0; i < mceLines.size(); ++i )
		{
			QStringList mceLine = mceLines.at( i ).split( '=' );
			for ( int j = 0; j < storedMceLines.size(); ++j )
			{
				QStringList storedMceLine = storedMceLines.at( j ).split( '=' );
				if ( storedMceLine.at( 0 ).trimmed() == mceLine.at( 0 ).trimmed() )
				{
					mceLinesNew.append( storedMceLines.at( j ) );
					paramFound = true;
				}
			}
			if ( !paramFound )
				mceLinesNew.append( mceLines.at( i ) );
			else
				paramFound = false;
		}
	}
	// using the datastorage exceptionally but this time for changing data in the external conf file
	if ( !iDataStorage->storeData( mceSection, mceLinesNew, McePath ) )
	{
		emit configuringError( DeviceManager::KeySettingsNotChanged );
		return false;
	}
	
	return true;
}

bool DeviceConfigurator::toggleInitScript( bool aEnable )
{
	QByteArray name;
	if( !whoAmI( name ) ) {
		emit configuringError( DeviceManager::InitScriptNotChanged );
		return false;
	}
	
	QString command = "update-rc.d ";
	QStringList empty;
	QByteArray result;
	
	if( name != "root" )
		command.prepend( "sudo " );
	
	if( aEnable )
		command.append( InitScript + " start 99 2" );
	else
		command.append( "-f " + InitScript + " remove" );
	
	if ( !systemIO( command, empty, result ) ) {
		emit configuringError( DeviceManager::InitScriptNotChanged );
		return false;
	}
	
	return true;
}

bool DeviceConfigurator::restartDevice()
{
	QString command = BinPath + DevStopper;
	QStringList args;
	QByteArray result;
	args.append( "restart" );
	if( !systemIO( command, args, result ) ) {
		emit configuringError( DeviceManager::DeviceNotRestarted );
		return false;
	}
	return true;
}

bool DeviceConfigurator::whoAmI( QByteArray &aName )
{
	QString command = "whoami";
	QStringList empty;
	if( !systemIO( command, empty, aName ) )
		return false;
	return true;
}

bool DeviceConfigurator::systemIO( const QString &aCommand, const QStringList &aArgs, QByteArray &aResult )
{
	qDebug() << "DeviceConfigurator::systemIO( QString &, QStringList &, QByteArray &)";
	qDebug() << "Command: " << aCommand;
	QProcess process;
	if( !aArgs.empty() )
		process.start( aCommand, aArgs );
	else
		process.start( aCommand );
	
	if( !process.waitForFinished() )
		return false;
	aResult = process.readAll();
	if( aResult.endsWith( '\n' ) )
		aResult.chop( 1 );
	
	qDebug() << "Result: " << aResult;
	
	return true;
}
