#ifndef ALARMSENDER_H_
#define ALARMSENDER_H_

extern "C"
{
#include <alarmd/alarm_event.h>
}
#include "DeviceDataStorage.h"

#include <QObject>
#include <QTime>

//! DeviceControl class. Sends and removes the alarms of the alarm daemon.
/*!
 * DeviceControl class. Sends and removes the alarms of the alarm daemon. The alarms make it possible
 * to rurn on and off the device in the desired points of time. This optional functionality is supported
 * in the "kiosk mode".
 */
class AlarmSender : public QObject
{
	Q_OBJECT

public:
	//! Constructor.
	/*!
	 * Constructor to initialize an AlarmSender instance
	 * \param aDataStorage Pointer to the DataStorage class instance.
	 */
	AlarmSender( DeviceDataStorage *aDataStorage );
	//! Destructor.
	virtual ~AlarmSender();

signals:
	//! Signal. Emitted if error occured during sending the alarms.
	/*!
	 * Signal. Emitted if error occured during sending the alarms.
	 * \param aErrorCode The error code.
	 * \param aAddInfo The additional error text.
	 */
	void alarmSendingFailed( DeviceManager::ErrorCode aErrorCode, QString &aAddInfo );

public slots:
	//! Slot. Consists and sends the alarms to the alarm daemon.
	/*!
	 * Slot. Consists and sends the alarms to the alarm daemon. Asks DeviceDataStorage to store the
	 * information about the sent alarms.
	 * \param aTurnOnAt The time the device is desired to be turned on.
	 * \param aTurnOffAt The time the device is desired to be turned off.
	 * \param aDays The days to indicate if the device is wanted to be turned on/off every day (7) or
	 * just weekdays (5).
	 */
	bool sendAlarms( QTime aTurnOnAt, QTime aTurnOffAt, int aDays = 5 );
	//! Slot. Removes the alarms from the alarm daemon.
	/*!
	 * Slot. Removes the alarms from the alarm daemon. Asks DeviceDataStorage to remove information
	 * about the sent alarms.
	 */
	void removeAlarms();
	//! Slot. Removes the alarms from the alarm daemon.
	/*!
	 * Slot. Removes the alarms from the alarm daemon. Asks DeviceDataStorage to fetch information about
	 * the stored alarms. After removing those alarms, asks DeviceDataStorage to remove information
	 * about the alarms.
	 */
	bool removeStoredAlarms();

private:
	//! Finds the first possible point of time to set the first alarms.
	/*!
	 * Finds the first possible point of time to set the first alarms.
	 * \param aDays The days to indicate if the device is wanted to be turned on/off every day (7) or
	 * just weekdays (5).
	 * \param aTime The time of the turning on/off alarm.
	 * \return The date of the first alarm.
	 */
	QDateTime findFirstTime( const int &aDays, const QTime &aTime );
	//! Counts days from the received date to the next working day.
	/*!
	 * Returns the day count from the received date to the next working day.
	 * \param aDateTime The date from where to count the days
	 * \return The amount of days
	 */
	int daysToNextWorkingDay( const QDateTime &aDateTime );
	//! Handles the result of an sent alarm.
	/*!
	 * Handles the result of an sent alarm. Updates the received aErrorText parameter in case the alarm
	 * sending fails. If not, it updates the internal list of succesfully sent alarms.
	 * \param aCookie The result of an sent alarm.
	 * \param aErrorText The result of an sent alarm.
	 * \return True if alarm sending succeeds; otherwise, false.
	 */
	bool handleCookie( cookie_t aCookie, QString &aErrorText );
	//! Maps the received aErrorCode parameter to a certain error message.
	/*!
	 * Maps the received aErrorCode parameter to a certain error message.
	 * \param aErrorCode The error code for the alarm sending failure.
	 * \return The mapped error message.
	 */
	QString mapError( alarm_error_t aErrorCode );

private:
	DeviceDataStorage *iDataStorage;
	QStringList iSentAlarms;

};

#endif /*ALARMSENDER_H_*/
