#include "AlarmSender.h"
#include "DeviceConstants.h"

#include <QtDebug>
#include <QFile>

AlarmSender::AlarmSender( DeviceDataStorage *aDataStorage )
{
	qDebug() << "AlarmSender( DeviceDataStorage * )";
	iDataStorage = aDataStorage;
}

AlarmSender::~AlarmSender()
{
	qDebug() << "~AlarmSender()";
}

bool AlarmSender::sendAlarms( QTime aTurnOnAt, QTime aTurnOffAt, int aDays )
{
	qDebug() << "AlarmSender::sendAlarms( QTime, QTime, int )";
	QString errortext;

	if ( aDays != 5 ) //if aDays is 5 we set the alarms only for working days
		aDays = 7;

	if ( !removeStoredAlarms() ) //remove possible old alarms
		return false;

	alarm_event_t eventOn;
	alarm_event_t eventOff;

	//zero the alarm events
	memset( &eventOn, 0, sizeof( alarm_event_t ) );
	memset( &eventOff, 0, sizeof( alarm_event_t ) );

	//set the common event fields for the alarm events
	eventOn.recurrence = 60 * 24 * 7; //minutes in a week
	eventOff.recurrence = 60 * 24 * 7; //minutes in a week
	eventOn.recurrence_count = -1;
	eventOff.recurrence_count = -1;
	eventOn.flags = ( ALARM_EVENT_BOOT | ALARM_EVENT_NO_DIALOG );
	eventOff.flags = ALARM_EVENT_NO_DIALOG;
	eventOff.exec_name = ( QString( BinPath + DevStopper ).toLatin1() ).data();;

	//find the first possible points of time to set the first alarms
	// Adjust the time. NOTE: alarms cannot be set to past
	QDateTime turnOnAtBasis = findFirstTime( aDays, aTurnOnAt );
	QDateTime turnOffAtBasis = findFirstTime( aDays, aTurnOffAt );

	cookie_t cookie;
	int index = 0;

	while ( index++ < aDays )
	{

		//send auto switch on alarms
		eventOn.alarm_time = turnOnAtBasis.toTime_t();
		cookie = alarm_event_add( &eventOn );
		if ( !handleCookie( cookie, errortext ) )
		{
			emit alarmSendingFailed( DeviceManager::NewAlarmsNotSent, errortext );
			removeAlarms();
			return false;
		}
		turnOnAtBasis = turnOnAtBasis.addDays( 1 ); //move on to next day
		if ( aDays == 5 ) //only for working days
			turnOnAtBasis = turnOnAtBasis.addDays( daysToNextWorkingDay( turnOnAtBasis ) );

		//send auto switch off alarms
		eventOff.alarm_time = turnOffAtBasis.toTime_t();
		cookie = alarm_event_add( &eventOff );
		if ( !handleCookie( cookie, errortext ) )
		{
			emit alarmSendingFailed( DeviceManager::NewAlarmsNotSent, errortext );
			removeAlarms();
			return false;
		}
		turnOffAtBasis = turnOffAtBasis.addDays( 1 ); //move on to next day
		if ( aDays == 5 ) //only for working days
			turnOffAtBasis = turnOffAtBasis.addDays( daysToNextWorkingDay( turnOffAtBasis ) );
	}
	if ( !iDataStorage->storeData( iDataStorage->dataSectionToString( DeviceDataStorage::Alarms ), iSentAlarms ) )
	{
		emit alarmSendingFailed( DeviceManager::NewAlarmsNotStored, errortext );
		removeAlarms();
		return false;
	}
	return true;
}

bool AlarmSender::removeStoredAlarms()
{
	qDebug() << "AlarmSender::removeStoredAlarms()";
	QStringList storedAlarms;
	QString errortext;
	if ( !iDataStorage->readData( iDataStorage->dataSectionToString( DeviceDataStorage::Alarms ), storedAlarms ) )
	{
		emit alarmSendingFailed( DeviceManager::OldAlarmsNotRemoved, errortext );
		return false;
	}

	for ( int i = 0; i < storedAlarms.size(); ++i )
		alarm_event_del( storedAlarms.at( i ).toLong() );
	
	// clearing the removed alarms from the storage by saving an empty alarm list
	storedAlarms.clear();
	iDataStorage->storeData( iDataStorage->dataSectionToString( DeviceDataStorage::Alarms ), storedAlarms );
	// catching an error in this case is not needed

	return true;
}

void AlarmSender::removeAlarms()
{
	qDebug() << "AlarmSender::removeAlarms()";
	for ( int i = 0; i < iSentAlarms.size(); ++i )
		alarm_event_del( iSentAlarms.at( i ).toLong() );

	// clearing the removed alarms from the storage by saving an empty alarm list
	iSentAlarms.clear();
	iDataStorage->storeData( iDataStorage->dataSectionToString( DeviceDataStorage::Alarms ), iSentAlarms );
	// catching an error in this case is not needed
}

QDateTime AlarmSender::findFirstTime( const int &aDays, const QTime &aTime )
{
	qDebug() << "AlarmSender::findFirstTime( const int &, const QTime & )";
	QDateTime currentTime = QDateTime::currentDateTime();
	QDateTime dateTimeBasis = QDateTime::currentDateTime();
	dateTimeBasis = dateTimeBasis.addSecs( dateTimeBasis.time().secsTo( aTime ) );

	if ( currentTime > dateTimeBasis )   //have to add a day
	{
		dateTimeBasis = dateTimeBasis.addDays( 1 );
	}
	if ( aDays == 5 )   //only for working days
	{
		dateTimeBasis = dateTimeBasis.addDays( daysToNextWorkingDay( dateTimeBasis ) );
	}
	return dateTimeBasis;
}

int AlarmSender::daysToNextWorkingDay( const QDateTime &aDateTime )
{
	qDebug() << "AlarmSender::daysToNextWorkingDay( const QDateTime & )";
	QString day = aDateTime.toString( "ddd" );
	if ( day == "Sat" )
		return 2;
	else if ( day == "Sun" )
		return 1;
	else
		return 0;
}

bool AlarmSender::handleCookie( cookie_t aCookie, QString &aErrorText )
{
	qDebug() << "AlarmSender::handleCookie( cookie_t, QStringList &, QString & )";
	if ( aCookie == 0 )
	{
		aErrorText = mapError( alarmd_get_error() );
		return false;
	}
	else
	{
		QString cookieStr;
		cookieStr.append( QString( "%1" ).arg( aCookie ) );
		iSentAlarms.append( cookieStr );
	}
	return true;
}

QString AlarmSender::mapError( alarm_error_t aErrorCode )
{
	qDebug() << "AlarmSender::mapError( alarm_error_t )";
	QString errorMessage;
	switch ( aErrorCode )
	{
		case ALARMD_SUCCESS:
			errorMessage = "No reason found."; //this should never happen
			break;
		case ALARMD_ERROR_DBUS:
			errorMessage = "An error with D-Bus occurred, probably coudn't get a D-Bus connection.";
			break;
		case ALARMD_ERROR_CONNECTION:
			errorMessage = "Could not contact alarmd via D-Bus.";
			break;
		case ALARMD_ERROR_INTERNAL:
			errorMessage = "Some alarmd or libalarm internal error, possibly a version mismatch.";
			break;
		case ALARMD_ERROR_MEMORY:
			errorMessage = "A memory allocation failed.";
			break;
		case ALARMD_ERROR_ARGUMENT:
			errorMessage = "An argument given by caller was invalid.";
			break;
		default: /* == 6 */
			errorMessage = "Alarm daemon not running.";
	}
	return errorMessage;
}
