#include "Communication.h"

#include <QList>
#include "ConnectionSettings.h"
#include "Meeting.h"
#include "Room.h"

// temporarly here
#include "Configuration.h"
#include <QDateTime>
#include <QDomDocument>
#include <QFile>
#include <QtDebug>

Communication::Communication( ConnectionSettings *aConnection )
{
	this->iConnectionSettings = new ConnectionSettings( *aConnection );

	this->iHttp = new QHttp( iConnectionSettings->serverUrl().toString(), QHttp::ConnectionModeHttps );
	this->iResponse = new QByteArray();

	connect( iHttp, SIGNAL( done( bool ) ), this, SLOT( notifyDone( bool ) ) );
	connect( iHttp, SIGNAL( readyRead( const QHttpResponseHeader& ) ), this, SLOT( processResponse( const QHttpResponseHeader& ) ) );
	connect( iHttp, SIGNAL( requestFinished( int, bool ) ), this, SLOT( handleResults( int, bool ) ) );
	connect( iHttp, SIGNAL( authenticationRequired( const QString&, quint16, QAuthenticator* ) ), this, SLOT( handleAuthentication( const QString&, quint16, QAuthenticator* ) ) );
	connect( iHttp, SIGNAL( sslErrors( const QList<QSslError>& ) ), iHttp, SLOT( ignoreSslErrors() )/*this, SLOT( notifySsl( const QList<QSslError>& ) )*/ );
	connect( iHttp, SIGNAL( stateChanged( int ) ), this, SLOT( notifyStateChange( int ) ) );
}

Communication::~Communication()
{
	if ( iConnectionSettings != 0 )
	{
		delete iConnectionSettings;
		iConnectionSettings = 0;
	}
	if ( iHttp != 0 )
	{
		delete iHttp;
		iHttp = 0;
	}
	if ( iResponse != 0 )
	{
		delete iResponse;
		iResponse = 0;
	}

}

void Communication::fetchMeetings( const QDateTime &aFrom, const QDateTime &aUntil, Room *aIn )
{
	// TODO : change dummy to correct implementation
	/*QList<Meeting*> meetings = readDummyData();

	for( int i=0; i<meetings.count(); i++ )
	{
		qDebug( "MEETING IN ARRAY AT %d : subject=%s ", i, meetings[i]->subject().toStdString().data() );
		qDebug( "start: %s", meetings[i]->startsAt().toString( "yyyy.MM.dd. hh:mm" ).toStdString().data() );
		qDebug( "end: %s", meetings[i]->endsAt().toString( "yyyy.MM.dd. hh:mm" ).toStdString().data() );
	}

	emit meetingsFetched( meetings );*/

	//Domain might not be necessary
	QString domain( "tieto-x" );

	if ( !iResponse->isEmpty() )
	{
		iResponse->clear();
	}

	if ( !iHttp )
	{
		iHttp = new QHttp( iConnectionSettings->serverUrl().toString(), QHttp::ConnectionModeHttps );

		connect( iHttp, SIGNAL( done( bool ) ), this, SLOT( notifyDone( bool ) ) );
		connect( iHttp, SIGNAL( readyRead( const QHttpResponseHeader& ) ), this, SLOT( processResponse( const QHttpResponseHeader& ) ) );
		connect( iHttp, SIGNAL( requestFinished( int, bool ) ), this, SLOT( handleResults( int, bool ) ) );
		connect( iHttp, SIGNAL( authenticationRequired( const QString&, quint16, QAuthenticator* ) ), this, SLOT( handleAuthentication( const QString&, quint16, QAuthenticator* ) ) );
		connect( iHttp, SIGNAL( sslErrors( const QList<QSslError>& ) ), iHttp, SLOT( ignoreSslErrors() ) );

	}

	iHttp->setUser( QString( domain + "\\" + iConnectionSettings->username() ), iConnectionSettings->password() );

	//QString credentials( domain + "\\" + user + ":" + passwd );
	QHttpRequestHeader header( QString( "POST" ), QString( "/ews/exchange.asmx" ) );

	header.setValue( "Host", iConnectionSettings->serverUrl().toString() );
	header.setContentType( "text/xml; charset=utf-8; action=\"http://schemas.microsoft.com/exchange/services/2006/messages/GetUserAvailability\"" );

	QByteArray content;

	//content.append( "<?xml version=\"1.0\" encoding=\"utf-8\"?>" );
	content.append( "<soap:Envelope xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:t=\"http://schemas.microsoft.com/exchange/services/2006/types\">" );
	content.append( "<soap:Body>" );
	content.append( "<GetUserAvailabilityRequest xmlns=\"http://schemas.microsoft.com/exchange/services/2006/messages\" xmlns:t=\"http://schemas.microsoft.com/exchange/services/2006/types\">" );
	content.append( "<t:TimeZone xmlns=\"http://schemas.microsoft.com/exchange/services/2006/types\">" );
	content.append( "<Bias>480</Bias><StandardTime><Bias>0</Bias><Time>02:00:00</Time><DayOrder>5</DayOrder><Month>10</Month><DayOfWeek>Sunday</DayOfWeek></StandardTime>" );
	content.append( "<DaylightTime><Bias>-60</Bias><Time>02:00:00</Time><DayOrder>1</DayOrder><Month>4</Month><DayOfWeek>Sunday</DayOfWeek></DaylightTime>" );
	content.append( "</t:TimeZone>" );
	content.append( "<MailboxDataArray>" );
	content.append( "<t:MailboxData><t:Email><t:Address>" );
	//ROOM ADDRESS
	content.append( aIn->address() );
	content.append( "</t:Address></t:Email><t:AttendeeType>Required</t:AttendeeType><t:ExcludeConflicts>false</t:ExcludeConflicts></t:MailboxData>" );
	content.append( "</MailboxDataArray>" );
	content.append( "<t:FreeBusyViewOptions><t:TimeWindow>" );
	content.append( "<t:StartTime>" );
	//FROM
	content.append( aFrom.toString( Qt::ISODate ) );
	//2009-04-13T00:00:00
	content.append( "</t:StartTime>" );
	content.append( "<t:EndTime>" );
	//TO
	content.append( aUntil.toString( Qt::ISODate ) );
	//2009-04-17T23:59:59
	content.append( "</t:EndTime>" );
	content.append( "</t:TimeWindow><t:MergedFreeBusyIntervalInMinutes>60</t:MergedFreeBusyIntervalInMinutes><t:RequestedView>DetailedMerged</t:RequestedView></t:FreeBusyViewOptions>" );
	content.append( "</GetUserAvailabilityRequest>" );
	content.append( "</soap:Body>" );
	content.append( "</soap:Envelope>" );

	iHttp->request( header, content );
}

//communication-related, rearrange if needed
void Communication::notifyStateChange( int aState )
{
	qDebug( "Communication::notifyStateChange to: %d", aState );
}

void Communication::processResponse( const QHttpResponseHeader& aHeader )
{
	qDebug( "CommunicationManager::processResponse : statuscode= %d", aHeader.statusCode() );
	if ( aHeader.statusCode() == 200 )
	{
		//Read data
		iResponse->append( iHttp->readAll() );
	}
	else
	{
		qDebug( "CommunicationManager::processResponse : some error occured" );
		qDebug( "::REASON: %s", aHeader.reasonPhrase().toStdString().data() );
	}
}
void Communication::notifyDone( bool aError )
{
	if ( !aError )
	{
		QList<Meeting*> meetings;
		if ( readMeetingDataFromResponse( meetings ) )
		{
			emit meetingsFetched( meetings );
		}
		else
		{
			//error occured
			qDebug( "CommunicationManager::notifyDone : some error occured in reading the meetings" );
		}
	}
	else
	{
		//error occured
		qDebug( "CommunicationManager::notifyDone : error received" );
	}
}
void Communication::handleResults( int /*aId*/, bool aError )
{
	if ( iHttp && aError )
	{
		switch ( iHttp->error() )
		{
			case QHttp::NoError:
				qDebug( "Communication::handleResults : Success" );
				break;
			case QHttp::HostNotFound:
				qDebug( "Communication::handleResults : The host name lookup failed" );
				break;
			case QHttp::ConnectionRefused:
				qDebug( "Communication::handleResults : The server refused the connection" );
				break;
			case QHttp::UnexpectedClose:
				qDebug( "Communication::handleResults : The server closed the connection unexpectedly" );
				break;
			case QHttp::InvalidResponseHeader:
				qDebug( "Communication::handleResults : The server sent an invalid response header" );
				break;
			case QHttp::WrongContentLength:
				qDebug( "Communication::handleResults : The client could not read the content correctly because an error with respect to the content length occurred" );
				break;
			case QHttp::Aborted:
				qDebug( "Communication::handleResults : The request was aborted with abort()" );
				break;
			case QHttp::ProxyAuthenticationRequiredError:
				qDebug( "Communication::handleResults : QHttp is using a proxy, and the proxy server requires authentication to establish a connection" );
				break;
			case QHttp::AuthenticationRequiredError:
				qDebug( "Communication::handleResults : The web server requires authentication to complete the request" );
				break;
			case QHttp::UnknownError:
				qDebug( "Communication::handleResults : An unknown error occurred" );
				break;
			default:

				break;
		}
	}

}
void Communication::handleAuthentication( const QString& /*aHost*/, quint16 /*aPort*/, QAuthenticator* aAuthenticator )
{
	qDebug( "Communication::handleAuthentication" );
	aAuthenticator->setPassword( iConnectionSettings->password() );
	aAuthenticator->setUser( iConnectionSettings->username() );
}
bool Communication::readMeetingDataFromResponse( QList<Meeting*>& aMeetings )
{

	QDomDocument doc;
	if ( !doc.setContent( *iResponse, true ) )
	{
		qDebug( "Communication::readMeetingDataFromResponse : some error occured" );
		return false;
	}
	QDomElement root = doc.documentElement();
	if ( root.tagName() != "Envelope" )
	{
		qDebug( "Communication::readMeetingDataFromResponse : some error occured" );
		return false;
	}

	QDomNodeList list = doc.elementsByTagName( QString( "CalendarEvent" ) );

	for ( int i = 0; i < list.count(); i++ )
	{
		QDomNode node = list.item( i );
		QDomElement e = node.toElement();

		if ( !e.isNull() && e.tagName().toLower() == "calendarevent" )
		{

			QString startsAt, endsAt, organizer, subject;

			for ( QDomNode subnode = node.firstChild(); !subnode.isNull(); subnode = subnode.nextSibling() )
			{
				QDomElement e = subnode.toElement();
				QString tagName = e.tagName().toLower();

				if ( tagName == QString( "starttime" ) )
				{
					startsAt = e.text();
				}
				else if ( tagName == QString( "endtime" ) )
				{
					endsAt = e.text();
				}
				/*else if( tagName == QString( "organizer" ) )
				{
					organizer = e.text();
				}*/
				else if ( tagName == QString( "calendareventdetails" ) )
				{
					for ( QDomNode detailnode = subnode.firstChild(); !detailnode.isNull(); detailnode = detailnode.nextSibling() )
					{
						QDomElement e = detailnode.toElement();
						QString tagName = e.tagName().toLower();

						if ( tagName == QString( "subject" ) )
						{
							subject = e.text();
						}
					}
				}
			}

			//TODO: Fix this: Timezone is currently -10hrs off
			QDateTime start = QDateTime::fromString( startsAt, Qt::ISODate ).addSecs( 36000 );
			QDateTime end = QDateTime::fromString( endsAt, Qt::ISODate ).addSecs( 36000 );

			//Organizer and description cannot be fetched yet
			aMeetings.append( new Meeting(
							0,
							Configuration::instance()->defaultRoom(),
							start,
							end,
							QString( "" ),
							QString( "" ),
							subject,
							QString( "" ) ) );

		}
	}
	return true;
}
