#ifndef MEETING_H_
#define MEETING_H_

#include <QString>
#include <QDateTime>

class Room;

//! Domain class. Describe appointments on Microsoft Exchange Server 2007.
/*!
 * Domain class. Describe appointments on Microsoft Exchange Server 2007.
 */
class Meeting
{
public:
	//! Constructor.
	/*!
	 * Constructor to initialize a Meeting instance.
	 * \param aRoom Pointer to the Room instance where the meeting will be held.
	 * \param aStartsAt The date and time when the meeting starts.
	 * \param aEndsAt The date and time when the meeting ends.
	 * \param aOrganizer The e-mail address of the organizer.
	 * \param aSubject The subject of the meeting.
	 */
	Meeting( int aPrimaryId, Room *aRoom, QDateTime aStartsAt, QDateTime aEndsAt, const QString &aOrganizerName = "", const QString &aOrganizerEMail = "", const QString &aSubject = "", const QString &aDescription = "" );
	//! Destructor.
	virtual ~Meeting();
	//! Gets the primary identifier of the meeting.
	/*!
	 * Gets the primary identifier of the meeting. The primary ID comes from the availability service
	 * which is used in Communication module to fetch the meetings at first. Secondary ID is needed to
	 * get more details of the meeting when user wants to open MeetingInfoDialog by clicking on cerain
	 * meeting in the schedule.
	 * \return Primary ID might be provided by the Microsoft Exchange Availability Service.
	 */
	int primaryId();
	//! Gets the secondary identifier of the meeting.
	/*!
	 * Gets the secondary identifier of the meeting. The primary ID comes from the availability service
	 * which is used in Communication module to fetch the meetings at first. Secondary ID is needed to
	 * get more details of the meeting when user wants to open MeetingInfoDialog by clicking on cerain
	 * meeting in the schedule.
	 * \return Secondary ID is provided by the translating the Primary one to Secondaty.
	 */
	int secondaryId();
	//! Gets the room where the meeting is held.
	/*!
	 * Gets the Room instance which identified the location of the meeting.
	 * \return Pointer to the Room instance.
	 */
	Room* room();
	//! Gets the organizer.
	/*!
	 * Gets formatted string which contains the organizer's name and mail address.
	 * \return The organizer's name and mail address.
	 */
	QString organizer();
	//! Gets the date and time when the meeting starts.
	/*!
	 * Gets the date and time when the meeting starts.
	 * \return The date and time.
	 */
	QDateTime startsAt();
	//! Gets the date and time when the meeting ends.
	/*!
	 * Gets the date and time when the meeting ends.
	 * \return The date and time.
	 */
	QDateTime endsAt();
	//! Gets the subject of the meeting.
	/*!
	 * Gets the subject of the meeting.
	 * \return The description of the meeting's subject.
	 */
	QString subject();
	//! Gets the description of the meeting.
	/*!
	 * Gets the description of the meeting. This field is a detailed description of the topic of the
	 * meeting provided by the meeting organizer. Note it can contain confidential information.
	 * \return The description of the meeting.
	 */
	QString description();
	//! Indicates if all the details of the current Meeting instance are available or not.
	/*!
	 * Indicates if all the details of the current Meeting instance are available or not. If yes, then
	 * Secondary ID is valid, otherwise not.
	 * \return TRUE if yes; otherwise, false.
	 */
	bool detailsAvailable();

	//! Sets the secondary Id of the current meeting.
	/*!
	 * Sets the secondary Id of the current meeting.
	 * \param aSecondaryId The secondary ID provided by translating to Primary into Secondary.
	 */
	void setSecondaryId( int aSecondaryId );
	//! Sets the organizer of the meeting.
	/*!
	 * Sets the organizer of the meeting.
	 * \param aOrganizerName The name of the organizer.
	 * \param aOrganizerEMail The e-mail address of the organizer.
	 */
	void setOrganizer( const QString &aOrganizerName, const QString &aOrganizerEMail );
	//! Sets the date and time when the meeting starts.
	/*!
	 * Sets the date and time when the meeting starts.
	 * \param aNewStart The new date and time.
	 */
	void setStartsAt( QDateTime aNewStart );
	//! Sets the date and time when the meeting endss.
	/*!
	 * Sets the date and time when the meeting endss.
	 * \param aNewEnd The new date and time.
	 */
	void setEndsAt( QDateTime aNewEnd );
	//! Sets new subject to the meeting.
	/*!
	 * Sets new subject to the meeting.
	 * \param aSubject The description of the new subject.
	 */
	void setSubject( const QString &aSubject );
	//! Sets new description to the meeting.
	/*!
	 * Sets new description to the meeting.
	 * \param aDescription The description of the meeting.
	 */
	void setDescription( const QString &aDescription );
	//! Checks if two objects are equal.
	/*!
	 * Checks if the another same type object is equal to the current instance.
	 * \param *aOther The pointer to another Meeting class instance.
	 * \return TRUE if equals; otherwise, FALSE.
	 */
	bool equals( Meeting *aOther );
	//! Checks if two Meetings are overlaping.
	/*!
	 * Checks if the current Meeting instance overlaps the parameter one.
	 * \param *aOther The pointer to another Meeting class instance.
	 * \return TRUE if overlaps; otherwise, FALSE.
	 */
	bool overlaps( Meeting* aOther );
	//! Makes a string to identify a meeting.
	/*!
	 * Makes string representation of the current Meeting instance.
	 * \return The string.
	 */
	QString toString();

private:
	int iPrimaryId;
	int iSecondaryId;
	Room *iRoom;
	QDateTime iStartsAt;
	QDateTime iEndsAt;
	QString iOrganizerName;
	QString iOrganizerEMail;
	QString iSubject;
	QString iDescription;
	bool iDetailsAvailable;

};

#endif /*MEETING_H_*/
