#ifndef DISPLAYSETTINGS_H_
#define DISPLAYSETTINGS_H_

#include <QDateTime>

//! Domain class. Stores display settings.
/*!
 * Domain class. Stores display settings such as date and time format, number of days shown
 * in week calendar.
 */
class DisplaySettings
{
public:
	//! Enumeration of days shown in week calendar.
	/*!
	 * Enumeration of days shown in week calendar.
	 */
	enum DaysInSchedule
	{
		WeekdaysInSchedule = 5,  /*!< Display days from Monday till Friday. */
		WholeWeekInSchedule = 7 /*!< Display days from Monday till Sunday. */
	};

	//! Enumeration of used date format.
	/*!
	 * Enumeration of used date format.
	 */
	enum DateFormat
	{
		LongDateFormat, /*!< Date displayed like "Monday 6 April 2009". */
		ShortDateFormat /*!< Date displayed like "Mon 6 Apr". */
	};

	//! Enumeration of used time format.
	/*!
	 * Enumeration of used time format.
	 */
	enum TimeFormat
	{
		TwelveHoursTimeFormat, /*!< Time displayed like "01:34 pm". */
		TwentyFourHoursTimeFormat /*!< Time displayed like "13:34". */
	};

public:
	//! Constructor
	/*!
	 * Constuctor to initialize DisplaySettings instance.
	 * \param aDateFormat DateFormat type variable to eliminate the date format in use.
	 * \param aTimeFormat TimeFormat type variable to eliminate the time in use.
	 * \param aDaysInSchedule Indicates how many days are shown in the schedule.
	 * \param aDayStartsAt Time which the schedule starts with.
	 * \param aDayEndsAt Time which the schedule ends with.
	 * \param aScreensaver Minutes for timer to launch screensaver.
	 */
	DisplaySettings( DateFormat aDateFormat, TimeFormat aTimeFormat, DaysInSchedule aDaysInSchedule, QTime aDayStartsAt, QTime aDayEndsAt, int aScreensaver );
	//! Destructor.
	virtual ~DisplaySettings();


	//! Gets date format.
	/*!
	 * Gets date format.
	 * \return Format string to display date.
	 */
	QString dateFormat();
	//! Gets time format.
	/*!
	 * Gets time format.
	 * \return Format string to display time.
	 */
	QString timeFormat();
	//! Gets number of days in week calendar.
	/*!
	 * Gets number of days in week calendar.
	 * \return Number of days to be shown.
	 */
	DaysInSchedule daysInSchedule();
	//! Gets first hour to be shown in calendar.
	/*!
	 * Gets first hour to be shown in calendar.
	 * \return First visible hour in calendar.
	 */
	QTime dayStartsAt();
	//! Gets last hour to be shown in calendar.
	/*!
	 * Gets last hour to be shown in calendar.
	 * \return Last visible hour in calendar.
	 */
	QTime dayEndsAt();

	//! Gets minutes to wait before screensaver launches.
	/*!
	 * Gets minutes to wait before screensaver launches.
	 * \return Waiting time in minutes.
	 */
	int screensaver();

	//! Sets date format.
	/*!
	 * Sets date format.
	 * \param aDateFormat Enumeration of date format.
	 */
	void setDateFormat( DateFormat aDateFormat );
	//! Sets time format.
	/*!
	 * Sets time format.
	 * \param aTimeFormat Enumeration of time format.
	 */
	void setTimeFormat( TimeFormat aTimeFormat );
	//! Sets number of days in week calendar.
	/*!
	 * Sets number of days in week calendar.
	 * \param aDaysInSchedule Number of days to be shown.
	 */
	void setDaysInSchedule( DaysInSchedule aDaysInSchedule );
	//! Sets first hour to be shown in calendar.
	/*!
	 * Sets first hour to be shown in calendar.
	 * \param aDayStartsAt First visible hour in calendar.
	 */
	void setDayStartsAt( QTime aDayStartsAt );
	//! Sets last hour to be shown in calendar.
	/*!
	 * Sets last hour to be shown in calendar.
	 * \param aDayEndsAt Last visible hour in calendar.
	 */
	void setDayEndsAt( QTime aDayEndsAt );
	//! Sets minutes to wait before screensaver launches.
	/*!
	 * Sets minutes to wait before screensaver launches.
	 * \param aWaitTime Waiting time as minutes.
	 */
	void setScreensaver( int aWaitTime = 1 );

private:
	DateFormat iDateFormat;
	TimeFormat iTimeFormat;
	DaysInSchedule iDaysInSchedule;
	QTime iDayStartsAt;
	QTime iDayEndsAt;
	int iScreensaver;

};

#endif /*DISPLAYSETTINGS_H_*/
