#ifndef CONNECTION_H_
#define CONNECTION_H_

#include <QString>
#include <QUrl>

//! Domain class. Stores connection information.
/*!
 * Domain class. Stores connection information like server url, username, password and refresh
 * interval.
 */
class ConnectionSettings
{
public:
	//! Constructor.
	/*!
	 * Overloaded. Constructor to initialize a ConnectionSettings instance.
	 * \param aServerUrl The URL of the Exchange server.
	 * \param aUsername The username used to log in to the server.
	 * \param aPassword The password used to log in to the server.
	 * \param aRefreshInterval The interval defines how often the server's data must be fetched. (Interpreted in seconds.)
	 */
	ConnectionSettings( const QUrl &aServerUrl, const QString &aUsername, const QString &aPassword, unsigned int aRefreshInterval );
	//! Copy constructor.
	/*!
	 * Overloaded. Constructor to initialize a ConnectionSettings instance.
	 * \param aOther ConnectionSettings class to be copied.
	 */
	ConnectionSettings( const ConnectionSettings &aOther );
	//! Destructor
	virtual ~ConnectionSettings();

	//! Gets the server address
	/*!
	 * Gets the server url address.
	 * \return Url address of server.
	 */
	QUrl serverUrl();
	//! Gets the username.
	/*!
	 * Gets username used to connect.
	 * \return Username.
	 */
	QString username();
	//! Gets the password.
	/*!
	 * Gets the password used to connect.
	 * \return Password.
	 */
	QString password();
	//! Gets the refresh interval.
	/*!
	 * Gets refresh interval in seconds to read the data from server.
	 * \return Refresh interval.
	 */
	unsigned int refreshInterval();

	//! Sets the server address
	/*!
	 * Sets the server url address.
	 * \param aServerUrl Url address of server.
	 */
	void setServerUrl( const QUrl &aServerUrl );
	//! Sets the username.
	/*!
	 * Sets username used to connect.
	 * \param aUsername Username.
	 */
	void setUsername( const QString &aUsername );
	//! Sets the password.
	/*!
	 * Sets the password used to connect.
	 * \param aPassword Password.
	 */
	void setPassword( const QString &aPassword );
	//! Sets the refresh interval.
	/*!
	 * Sets refresh interval in seconds to read the data from server.
	 * \param aRefreshInterval Refresh interval.
	 */
	void setRefreshInterval( unsigned int aRefreshInterval );


private:
	QUrl iServerUrl;
	QString iUsername;
	QString iPassword;
	unsigned int iRefreshInterval; // in seconds

};

#endif /*CONNECTION_H_*/
