#ifndef CONFIGURATION_H_
#define CONFIGURATION_H_

#include <QObject>
#include <QString>
#include <QList>
#include <QTime>

class ConnectionSettings;
class StartupSettings;
class DisplaySettings;
class Room;
class QDomNode;

//! Domain class. Store application wide configuration values.
/*!
 * Domain class. Store application wide configuration values. The values are read from a configuration
 * file at initialization time. Since there is one appliation per device normally running, therefore
 * there is only one instance of this class, which is accessible by using a statis getter method.
 */
class Configuration : public QObject
{
	Q_OBJECT

private:
	//! Constructor.
	/*!
	 * Constructor to initialize an Configuration instance. The method populates the object by reading
	 * through the application's configuration file.
	 */
	Configuration();

public:
	//! Destructor
	virtual ~Configuration();

	//! Static. Gets the application wide configuration instance.
	/*!
	 * Static Gets the static instance of this class. It is used to read the configuration information.
	 * \return Pointer to the Configuration instalce.
	 */
	static Configuration* instance();
	//! Gets the connection settings.
	/*!
	 * Gets the connection settings.
	 * \return Pointer to ConnectionSettings instance.
	 */
	ConnectionSettings* connectionSettings();
	//! Gets the detault room.
	/*!
	 * Gets the default meeting room.
	 * \return Pointer to the default room.
	 */
	Room* defaultRoom();
	//! Gets the language code.
	/*!
	 * Gets the language code.
	 * \return Language code in ISO 3166-1 alpha-2 format.
	 */
	QString languageCode();
	//! Gets the list of meeting rooms.
	/*!
	 * Gets the list of meeting rooms.
	 * \return List of rooms.
	 */
	QList<Room*> rooms();
	//! Gets startup settings.
	/*!
	 * Gets the startup settings.
	 * \return Pointer to StartupSettings instance.
	 */
	StartupSettings* startupSettings();
	//! Gets display settings.
	/*!
	 * Gets the display settings.
	 * \return Pointer to DisplaySettings instance.
	 */
	DisplaySettings* displaySettings();
	//! Gets the administrator's password.
	/*!
	 * Gets the administrator's password
	 * \return Administration password.
	 */
	QString adminPassword();
	//! Sets room list.
	/*!
	 * Sets  room list.
	 * \param aRooms List of rooms
	 */
	void setRooms( const QList<Room*> aRooms );
	//! Sets admin password.
	/*!
	 * Sets  admin password.
	 * \param aPassword admin password
	 */
	void setAdminPassword( const QString aPassword );

	//! Compares given password to the saved one.
	/*!
	 * Hashes aPassword and compares it with iAdminPassword.
	 * \param aPassword admin password
	 * \return True if hashed aPassword is equal with iAdminPassword
	 */
	bool compareAdminPassword( const QString aPassword );

public slots:

	//! Saves setting values to file.
	/*!
	 * Writes setting values to configuration file.
	 */
	void save();

private:
	//! Static. Reads the configuration instance from XML file.
	/*!
	 * Static. Reads the configuration information from configuration file.
	 * \param aPath path and name of configuration file
	 * \return Configuration object.
	 */
	static Configuration* readFromXML( const QString &aPath );
	//! Static. Reads settings of connection from and XML node.
	/*!
	 * Static. Reads settings of connection from an XML node.
	 * \param aXml QDomNode containing connection parameters.
	 * \return Pointer to ConnectionSettings object.
	 */
	static ConnectionSettings* readConnectionSettings( const QDomNode &aXML );
	//! Static. Reads rooms from an XML node.
	/*!
	 * Static. Reads rooms from an XML node.
	 * \param aXml QDomNode containing meeting room parameters
	 * \return List of meetingrooms.
	 */
	static QList<Room*> readRooms( const QDomNode &aXML );
	//! Static. Reads language code from an XML node.
	/*!
	 * Static. Reads rooms from an XML node.
	 * \param aXml QDomNode containing language code
	 * \return Language code.
	 */
	static QString readLanguageCode( const QDomNode &aXML );
	//! Static. Reads settings of startup from an XML node.
	/*!
	 * Static. Reads settings of startup from an XML node.
	 * \param aXml QDomNode containing startup parameters
	 * \return Pointer to the read StartupSettings object.
	 */
	static StartupSettings* readStartupSettings( const QDomNode &aXML );
	/*!
	 * Static function to load and store display settings from xml node.
	 * \param aXml QDomNode containing display parameters
	 * \return Pointer to the read DisplaySettings object.
	 */
	static DisplaySettings* readDisplaySettings( const QDomNode &aXML );
	//! Static. Reads adminstrator's password from an XML node.
	/*!
	 * Static. Reads adminstrator's password from an XML node.
	 * \param aXml QDomNode containing admin password
	 * \return Admin password.
	 */
	static QString readAdminPassword( const QDomNode &aXML );

	//! Saves connection data to the document.
	/*!
	 * Reads data from iConnectionSettings and saves it to the aXML document.
	 * \param aXml QDomNode containing connection parameters.
	 */
	void saveConnectionSettings( const QDomNode &aXML );
	//! Saves meeting rooms to the document.
	/*!
	 * Reads data from iRooms list and saves it to the aXML document.
	 * \param aXml QDomNode containing meeting room parameters
	 */
	void saveRooms( const QDomNode &aXML );
	//! Saves the language code.
	/*!
	 * Reads data from iLanguageCode and saves it to the aXML document.
	 * \param aXml QDomNode containing language code
	 */
	void saveLanguageCode( const QDomNode &aXML );
	//! Saves startup setting data to the document.
	/*!
	 * Reads data from iStartupSettings and saves it to the aXML document.
	 * \param aXml QDomNode containing startup parameters
	 */
	void saveStartupSettings( const QDomNode &aXML );
	/*!
	//! Saves display setting data to the document.
	/*!
	 * Reads data from iDisplaySettings and saves it to the aXML document.
	 * \param aXml QDomNode containing display parameters
	 */
	void saveDisplaySettings( const QDomNode &aXML );
	//! Saves admin password to the document.
	/*!
	 * Reads data from iAdminPassword and saves it to the aXML document.
	 * \param aXml QDomNode containing admin password
	 */
	void saveAdminPassword( const QDomNode &aXML );

	//! Hash password with md5 method.
	/*!
	 * Hash password with md5 method.
	 * \param aPassword password to be encoded
	 * \return Encoded password.
	 */
	QString hashPassword( const QString aPassword );

private:
	//! Path and name of configuration file
	static QString sConfigurationPath;
	//! The static instance which is in use to read the configuration.
	static Configuration *sInstance;
	//! Static constant to store the default interval used for connection refresh.
	static const unsigned int sDefaultInterval = 60;
	//! Pointer to the ConnectionSettings object
	ConnectionSettings *iConnectionSettings;
	//! Stores startup settings.
	StartupSettings *iStartupSettings;
	//! Stores display settings.
	DisplaySettings *iDisplaySettings;
	//! List of meeting rooms.
	QList<Room*> iRooms;
	//! Stores administrator password.
	QString iAdminPassword;
	//! Stores language code
	QString iLanguageCode;

};

#endif /*CONFIGURATION_H_*/
