#include "Configuration.h"
#include "ConnectionSettings.h"
#include "StartupSettings.h"
#include "DisplaySettings.h"
#include "Room.h"
#include <QDomDocument>
#include <QDomElement>
#include <QDomNode>
#include <QFile>
#include <QCryptographicHash>

#include <QtDebug>

Configuration * Configuration::sInstance = 0;
QString Configuration::sConfigurationPath = "/etc/QtMeetings.conf";

Configuration::Configuration() :
		iConnectionSettings( 0 ),
		iStartupSettings( 0 ),
		iDisplaySettings( 0 )
{
}

Configuration::~Configuration()
{
	delete iConnectionSettings;
	delete iStartupSettings;
	delete iDisplaySettings;
	while ( !iRooms.isEmpty() )
	{
		delete iRooms.takeFirst();
	}
}

Configuration* Configuration::instance()
{
	if ( sInstance == 0 )
	{
		sInstance = readFromXML( sConfigurationPath );
	}
	return sInstance;
}

ConnectionSettings* Configuration::connectionSettings()
{
	return iConnectionSettings;
}

StartupSettings * Configuration::startupSettings()
{
	return iStartupSettings;
}

DisplaySettings * Configuration::displaySettings()
{
	return iDisplaySettings;
}

Room* Configuration::defaultRoom()
{
	// default room is stored as the first element of the list
	return ( iRooms.count() == 0 ) ? 0 : iRooms.at( 0 );
}

QString Configuration::languageCode()
{
	return iLanguageCode;
}

QList<Room*> Configuration::rooms()
{
	return iRooms;
}

QString Configuration::adminPassword()
{
	return iAdminPassword;
}

void Configuration::save()
{
	QDomDocument doc;
	QFile file( sConfigurationPath );

	if ( !file.open( QIODevice::ReadWrite ) )
	{
		return;
	}

	if ( !doc.setContent( &file ) )
	{
		file.close();
		return;
	}

	QDomElement root = doc.documentElement();
	// Save all attributes to document

	saveAdminPassword( root );

	for ( QDomNode node = root.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "connection" ) )
		{
			saveConnectionSettings( node );
		}
		else if ( tagName == QString( "rooms" ) )
		{
			saveRooms( node );
		}
		else if ( tagName == QString( "language" ) )
		{
			saveLanguageCode( node );
		}
		else if ( tagName == QString( "startup" ) )
		{
			saveStartupSettings( node );
		}
		else if ( tagName == QString( "display" ) )
		{
			saveDisplaySettings( node );
		}
	}

	//! Empty the file from previous content and write again with new one
	file.resize( 0 );
	file.write( doc.toByteArray( 4 ) );	//! 4 as intent

	file.close();

}

void Configuration::saveConnectionSettings( const QDomNode &aXML )
{
	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "serverurl" ) )
		{
			QDomText t = node.ownerDocument().createTextNode( iConnectionSettings->serverUrl().toString() );
			e.replaceChild( t, e.firstChild() );
		}
		else if ( tagName == QString( "username" ) )
		{
			QDomText t = node.ownerDocument().createTextNode( iConnectionSettings->username() );
			e.replaceChild( t, e.firstChild() );
		}
		else if ( tagName == QString( "password" ) )
		{
			QDomText t = node.ownerDocument().createTextNode( iConnectionSettings->password() );
			e.replaceChild( t, e.firstChild() );
		}
		else if ( tagName == QString( "refreshinterval" ) )
		{
			QString s = QString( "%1" ).arg( iConnectionSettings->refreshInterval() );
			QDomText t = node.ownerDocument().createTextNode( s );
			e.replaceChild( t, e.firstChild() );
		}
	}
}

void Configuration::saveRooms( const QDomNode &aXML )
{
	//! List of rooms must be cleared and rewritten again
	QDomDocument doc = aXML.ownerDocument();
	QDomNode node = aXML.firstChild();
	QDomElement e = node.toElement();
	QString tagName = e.tagName().toLower();

	if ( tagName == QString( "room" ) )
	{
		e.clear();
	}
	else
	{
		// Should not get here
		return;
	}

	QList<Room*>::iterator i;
	for ( i = iRooms.begin(); i != iRooms.end(); ++i )
	{
		QDomElement tag = doc.createElement( "room" );
		node.appendChild( tag );

		// First room in the list is a dafault room
		if ( i == iRooms.begin() )
		{
			tag.setAttribute( "default", "true" );
		}

		QDomElement ename = doc.createElement( "name" );
		QDomText tname = node.ownerDocument().createTextNode(( *i )->name() );
		ename.appendChild( tname );
		tag.appendChild( ename );

		QDomElement eaddress = doc.createElement( "address" );
		QDomText taddress = node.ownerDocument().createTextNode(( *i )->address() );
		eaddress.appendChild( taddress );
		tag.appendChild( eaddress );
	}

}

void Configuration::saveLanguageCode( const QDomNode &aXML )
{
	QDomElement e = aXML.toElement();
	e.setAttribute( "code", languageCode() );
}

void Configuration::saveStartupSettings( const QDomNode &aXML )
{
	QDomElement e = aXML.toElement();

	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "powersaving" ) )
		{
			( iStartupSettings->isPowersavingEnabled() ?
			  e.setAttribute( "enabled", "true" ) :
			  e.setAttribute( "enabled", "false" ) );

			e.setAttribute( "on", iStartupSettings->turnOnAt().toString( "hh:mm" ) );
			e.setAttribute( "off", iStartupSettings->turnOffAt().toString( "hh:mm" ) );
		}
	}
}

void Configuration::saveDisplaySettings( const QDomNode &aXML )
{
	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "schedule" ) )
		{
			for ( QDomNode scheduleNode = node.firstChild(); !scheduleNode.isNull(); scheduleNode = scheduleNode.nextSibling() )
			{
				QDomElement scheduleElem = scheduleNode.toElement();
				tagName = scheduleElem.tagName().toLower();

				if ( tagName == QString( "week" ) )
				{
					scheduleElem.setAttribute( "days", iDisplaySettings->daysInSchedule() );
				}
				else if ( tagName == QString( "day" ) )
				{
					scheduleElem.setAttribute( "startsat", iDisplaySettings->dayStartsAt().toString( "hh:mm" ) );
					scheduleElem.setAttribute( "endsat", iDisplaySettings->dayEndsAt().toString( "hh:mm" ) );
				}
			}	// end of for
		}	// end of schedule
		else if ( tagName == QString( "dateformat" ) )
		{
			QDomText t = node.ownerDocument().createTextNode( iDisplaySettings->dateFormat() );
			e.replaceChild( t, e.firstChild() );
		}
		else if ( tagName == QString( "timeformat" ) )
		{
			QDomText t = node.ownerDocument().createTextNode( iDisplaySettings->timeFormat() );
			e.replaceChild( t, e.firstChild() );
		}
		else if ( tagName == QString( "screensaver" ) )
		{
			QString s = QString( "%1" ).arg( iDisplaySettings->screensaver() );
			QDomText t = node.ownerDocument().createTextNode( s );
			e.replaceChild( t, e.firstChild() );
		}
	}
}

void Configuration::saveAdminPassword( const QDomNode &aXML )
{
	QDomElement e = aXML.toElement();
	e.setAttribute( "password", adminPassword() );
}


Configuration* Configuration::readFromXML( const QString &aPath )
{
	QDomDocument doc;
	QFile file( aPath );

	if ( !file.open( QIODevice::ReadOnly ) )
	{
		return 0;
	}
	if ( !doc.setContent( &file ) )
	{
		file.close();
		return 0;
	}
	file.close();

	QDomElement root = doc.documentElement();
	// check if the file is the one we need
	if ( root.tagName().toLower() != QString( "configuration" ) )
	{
		return 0;
	}

	Configuration* conf = new Configuration();

	conf->iAdminPassword = Configuration::readAdminPassword( root );

	for ( QDomNode node = root.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "connection" ) )
		{
			conf->iConnectionSettings = Configuration::readConnectionSettings( node );
		}
		else if ( tagName == QString( "rooms" ) )
		{
			conf->iRooms = Configuration::readRooms( node );
		}
		else if ( tagName == QString( "language" ) )
		{
			conf->iLanguageCode = Configuration::readLanguageCode( node );
		}
		else if ( tagName == QString( "startup" ) )
		{
			conf->iStartupSettings = Configuration::readStartupSettings( node );
		}
		else if ( tagName == QString( "display" ) )
		{
			conf->iDisplaySettings = Configuration::readDisplaySettings( node );
		}
	}

	return conf;
}

ConnectionSettings* Configuration::readConnectionSettings( const QDomNode &aXML )
{
	QString server, username, password;
	unsigned int interval = Configuration::sDefaultInterval;

	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "serverurl" ) )
		{
			server = e.text();
		}
		else if ( tagName == QString( "username" ) )
		{
			username = e.text();
		}
		else if ( tagName == QString( "password" ) )
		{
			password = e.text();
		}
		else if ( tagName == QString( "refreshinterval" ) )
		{
			bool success = false;
			unsigned int intervalTMP = e.text().toUInt( &success );
			if ( success )
			{
				interval = intervalTMP;
			}
		}
	}

	return new ConnectionSettings( server, username, password, interval );
}

QList<Room*> Configuration::readRooms( const QDomNode &aXML )
{
	QList<Room*> rooms;

	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "room" ) )
		{
			QString name, address;

			for ( QDomNode roomNode = node.firstChild(); !roomNode.isNull(); roomNode = roomNode.nextSibling() )
			{
				QDomElement roomElem = roomNode.toElement();
				tagName = roomElem.tagName().toLower();
				if ( tagName == QString( "name" ) )
				{
					name = roomElem.text();
				}
				else if ( tagName == QString( "address" ) )
				{
					address = roomElem.text();
				}
			}
			Room* room = new Room( name, address );
			QString defaultAttr = e.attribute( "default" ).toLower();
			if ( defaultAttr == QString( "true" ) )
			{
				rooms.insert( 0, room );
			}
			else
			{
				rooms.append( room );
			}
		}
	}

	return rooms;
}

QString Configuration::readLanguageCode( const QDomNode &aXML )
{
	QDomElement e = aXML.toElement();
	QString tagName = e.tagName().toLower();

	if ( e.hasAttribute( "code" ) )
	{
		return e.attribute( "code" );
	}
	else
	{
		// default language is English
		return "EN";
	}
}

StartupSettings * Configuration::readStartupSettings( const QDomNode &aXML )
{
	QDomElement e = aXML.toElement();

	bool isPowersavingEnabled = false;
	QTime turnOnAt, turnOffAt;

	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "powersaving" ) )
		{
			if ( e.hasAttribute( "enabled" ) &&
				  e.attribute( "enabled" ) == QString( "true" ) )
			{
				isPowersavingEnabled = true;
			}

			if ( e.hasAttribute( "on" ) )
			{
				QString on = e.attribute( "on" );
				turnOnAt = QTime::fromString( on, "hh:mm" );

			}

			if ( e.hasAttribute( "off" ) )
			{
				QString off = e.attribute( "off" );
				turnOffAt = QTime::fromString( off, "hh:mm" );
			}
		}
	}

	return new StartupSettings( isPowersavingEnabled, turnOnAt, turnOffAt );
}

DisplaySettings * Configuration::readDisplaySettings( const QDomNode &aXML )
{
	DisplaySettings::DaysInSchedule daysInSchedule;
	QTime dayStartsAt, dayEndsAt;
	DisplaySettings::DateFormat dateformat;
	DisplaySettings::TimeFormat timeformat;
	int screensaver = 1;	//! Default value for screensaver wait time

	for ( QDomNode node = aXML.firstChild(); !node.isNull(); node = node.nextSibling() )
	{
		QDomElement e = node.toElement();
		QString tagName = e.tagName().toLower();

		if ( tagName == QString( "schedule" ) )
		{
			for ( QDomNode scheduleNode = node.firstChild(); !scheduleNode.isNull(); scheduleNode = scheduleNode.nextSibling() )
			{
				QDomElement scheduleElem = scheduleNode.toElement();
				tagName = scheduleElem.tagName().toLower();

				if ( tagName == QString( "week" ) )
				{
					if ( scheduleElem.hasAttribute( "days" ) )
					{
						// default value is 5, only other supported possibility is 7
						bool success = false;
						unsigned int days = scheduleElem.attribute( "days" ).toUInt( &success );
						daysInSchedule = ( success && days == 7 ) ? DisplaySettings::WholeWeekInSchedule : DisplaySettings::WeekdaysInSchedule;
					}

				}
				else if ( tagName == QString( "day" ) )
				{
					if ( scheduleElem.hasAttribute( "startsat" ) )
					{
						QString time = scheduleElem.attribute( "startsat" );
						dayStartsAt = QTime::fromString( time, "hh:mm" );
					}
					if ( scheduleElem.hasAttribute( "endsat" ) )
					{
						QString time = scheduleElem.attribute( "endsat" );
						dayEndsAt = QTime::fromString( time, "hh:mm" );
					}

				}
			}	// end of for
		}	// end of schedule
		else if ( tagName == QString( "dateformat" ) )
		{
			//! Not able to display long format anyway at the moment
			/*
			if ( e.text() == QObject::tr( "dddd d MMMM yyyy" ) )
				dateformat = DisplaySettings::LongDateFormat;
			else
				dateformat = DisplaySettings::ShortDateFormat;
			*/
			dateformat = DisplaySettings::ShortDateFormat;
		}
		else if ( tagName == QString( "timeformat" ) )
		{
			//! Not able to display "TwelveHoursTimeFormat" anyway at the moment
			/*
			if ( e.text() == QObject::tr( "hh:mm ap" ) )
				timeformat = DisplaySettings::TwelveHoursTimeFormat;
			else
				timeformat = DisplaySettings::TwentyFourHoursTimeFormat;
			*/
			timeformat = DisplaySettings::TwentyFourHoursTimeFormat;
		}
		else if ( tagName == QString( "screensaver" ) )
		{
			bool success = false;
			unsigned int screensaverTMP = e.text().toUInt( &success );
			if ( success )
			{
				screensaver = screensaverTMP;
			}
		}
	}

	return new DisplaySettings( dateformat, timeformat, daysInSchedule, dayStartsAt, dayEndsAt, screensaver );
}

QString Configuration::readAdminPassword( const QDomNode &aXML )
{
	QDomElement e = aXML.toElement();
	QString tagName = e.tagName().toLower();
	if ( e.hasAttribute( "password" ) )
	{
		// reading does NOT care if something is encrypted or not!
		return e.attribute( "password" );
	}
	else
	{
		return 0;
	}
}

void Configuration::setAdminPassword( const QString aPassword )
{
	iAdminPassword = hashPassword( aPassword );
}

bool Configuration::compareAdminPassword( const QString aPassword )
{
	return ( iAdminPassword == hashPassword( aPassword ) );
}

void Configuration::setRooms( const QList<Room*> aRooms )
{
	iRooms = aRooms;
}

QString Configuration::hashPassword( const QString aPassword )
{
	QCryptographicHash *hash = new QCryptographicHash( QCryptographicHash::Md5 );
	hash->addData( aPassword.toUtf8() );
	QByteArray password = hash->result();
	delete hash;

	return QString( password );
}
