#ifndef ENGINE_H_
#define ENGINE_H_

#include <QObject>
#include <QDateTime>
#include "Room.h"

class QTimer;
class Clock;
class Configuration;
class CommunicationManager;
class Meeting;
class DeviceManager;

//! BusinessLogic class. Contains all the business logic of the application.
/*!
 * BusinessLogic class. Contains all the business logic of the application. It is responsible
 * for connecting user interface to lower application layers (IO).
 */
class Engine : public QObject
{
	Q_OBJECT

public:
	//! Constructor.
	/*!
	 * Constructor to initialize an Engine instance.
	 */
	Engine();
	//! Destructor.
	virtual ~Engine();

	//! Gets the clock instance used by the object to get up-to-date date and time info.
	/*!
	 * Gets the clock instance used by the object to get up-to-date date and time info.
	 * \return Pointer to the Clock instance.
	 */
	Clock* clock();
	//! Gets the application's configuration.
	/*!
	 * Gets the application's configuration.
	 * \return Pointer to the Configuration instance.
	 */
	Configuration* configuration();
	//! Gets default room of the application.
	/*!
	 * Gets default room of the application.
	 * \return Pointer to the default Room instance.
	 */
	Room* defaultRoom();
	//! Gets the deviceManager instance
	/*!
	 * Gets the deviceManager instance.
	 * \return Pointer to the deviceManager instance.
	 */
	DeviceManager* deviceManager();

signals:
	//! Signal. Emitted if initialization of the current instance failed.
	/*!
	 * Signal. Emitted if initialization of the current instance failed, if reading of the configuration
	 * was not successful. It's purpose to inform the userinterface that the Engine is not ready for
	 * controlling the application, application must be shut down.
	 */
	void initializationFailed();
	//! Signal. Emitted if the availability information of the specified room changed.
	/*!
	 * Signal. Emitted if the availability information of the specified room changed.
	 * \param aRoom The Room instance which availability changed.
	 * \param aStatus The status of the room.
	 * \param aUntil Time until the spacified status is valid.
	 */
	void roomStatusChanged( Room *aRoom, Room::Status aStatus, QTime aUntil );
	//! Signal. Emitted if new meeting was found on the server.
	/*!
	 * Signal. Emitted if new meeting was found on the server.
	 * \param aMeeting The new meeting which was added.
	 */
	void meetingAdded( Meeting *aMeeting );
	//! Signal. Emitted if meeting was deleted on the server.
	/*!
	 * Signal. Emitted if meeting was deleted on the server.
	 * \param aMeeting The meeting which was deleted.
	 */
	void meetingDeleted( Meeting *aMeeting );
	//! Signal. Emitted error occured and error message must be shown on UI.
	/*!
	 * Signal. Emitted error occured and error message must be shown on UI.
	 * \param aErrorMessage The message.
	 */
	void error( const QString &aErrorMessage );

public slots:
	//! Slot. Checks actual availability information of the specified room.
	/*!
	 * Slot. Checks actual availability information of the specified room.
	 * \param aRoom The room which availability information is needed.
	 */
	void roomStatusInfoNeeded( Room *aRoom );
	//! Slot. Fetches meetings from the server.
	/*!
	 * Slot. Fetches meetings from the server, exact parameters are specified in the parameter list.
	 * \param aFrom Time from when the meetings need to be fetched.
	 * \param aUntil Time until when the meetings need to be fetched.
	 * \param aIn The room which meetings need to be fetched.
	 */
	void fetchMeetings( const QDateTime &aFrom, const QDateTime &aUntil, Room *aIn );
	//! Slot. Creates meeting by calling lower layer methods.
	/*!
	 * Slot. Creates meeting, i.e. requests an appointment by calling lower layer methods.
	 * \param aMeeting The meeting to be requested.
	 * \param aUsername The username of the individual, whose Exchange account requests the meeting from
	 * the server.
	 * \param aPassword The password which belongs to the username specified earlier.
	 */
	void createMeeting( const Meeting *aMeeting, const QString &aUsername, const QString &aPassword );
	//! Slot. Updates meeting by calling lower layer methods.
	/*!
	 * Slot. Updates meeting, i.e. requests an appointment update by calling lower layer methods.
	 * \param aMeeting The meeting to be updated.
	 * \param aUsername The username of the individual, whose Exchange account requests the update from
	 * the server.
	 * \param aPassword The password which belongs to the username specified earlier.
	 */
	void updateMeeting( const Meeting *aMeeting, const QString &aUsername, const QString &aPassword );
	//! Slot. Updates meeting by calling lower layer methods.
	/*!
	 * Slot. Updates meeting, i.e. requests an appointment update by calling lower layer methods.
	 * \param aMeeting The meeting to be updated.
	 * \param aUsername The username of the individual, whose Exchange account requests the update from
	 * the server.
	 * \param aPassword The password which belongs to the username specified earlier.
	 */
	void deleteMeeting( const Meeting *aMeeting, const QString &aUsername, const QString &aPassword );
	//! Slot. Sets the current meeting room.
	/*!
	 * Slot. Sets the current meeting room iCurrentRoom.
	 * \param aCurrentRoom
	 */
	void currentRoomChanged( Room *aCurrentRoom );

private slots:
	//! Slot. Handles errors.
	/*!
	 * Slot. Handles errors and informs the UI by emitting the error() signal with the message in
	 * parameter.
	 * \param aCode The error code.
	 */
	void errorHandler( int aCode );
	//! Slot. Fetches meetings from the server.
	/*!
	 * Slot. Fetches meetings from the server. Parameters are hard coded: the meetings of the default
	 * room from current and +/- 2 weeks are fetched.
	 */
	void fetchMeetings();
	//! Slot. Saves fetched meetings to the current instance's local storage.
	/*!
	 * Slot. Saves fetched meetings to the current instance's local storage. Meetings are soted in a
	 * private QList, it is iterated through and signals are emitted if there is any new, updated or
	 * deleted meeting.
	 * \param aMeetings The list of freshly fetched meetings.
	 */
	void meetingsFetched( QList<Meeting*> aMeetings );
	//! Slot. Checks the availability of all the rooms.
	/*!
	 * Slot. Checks the availability of all the rooms by iterating through the current object's local
	 * room storage and calling the roomStatusInfoNeeded() separately on each of them.
	 */
	void checkStatusOfAllRooms();

private:
	//! Provides the index of the Meeting instance which is at the specified time.
	/*!
	 * Provides the index of the Meeting instance which is at the specified time. If there are
	 * overlapping meetings then returns one of then undetetministically.
	 * \param aRoom The room which meetings are looked through.
	 * \param aAt Date and time when the meeting is already going.
	 * \return Index of the meeting if found; otherwise, -1.
	 */
	int indexOfMeetingAt( Room *aRoom, QDateTime aAt );
	//! Provides the index of the Meeting instance which is starts the closest to the specified time.
	/*!
	 * Provides the index of the Meeting instance which is starts the closest to the specified time.
	 * If there are overlapping meetings then returns one of then undetetministically.
	 * \param aRoom The room which meetings are looked through.
	 * \param aAt Date and time when the meeting is not yet started.
	 * \return Index of the meeting if found; otherwise, -1.
	 */
	int indexOfMeetingAfter( Room *aRoom, QDateTime aAfter );
	//! Indicates if the QList contains the Meeting or not.
	/*!
	 * Indicates if the QList contains the Meeting or not.
	 * \param aList List of meetings.
	 * \param aMeeting The meeting which is seeked in the list for.
	 * \return True if contains; otherwise, false.
	 */
	bool isMeetingInList( QList<Meeting*> aList, Meeting *aMeeting );

private:
	static QTime endOfTheDay;

	Clock *iClock;
	Configuration *iConfiguration;
	CommunicationManager *iCommunication;
	DeviceManager *iDevice;

	QTimer *iAutoRefresh;

	QList<Meeting*> iMeetings;

	Room *iCurrentRoom;	/*! Not owned */
};

#endif /*ENGINE_H_*/
