#include "Engine.h"

#include <QTimer>
#include <QList>
#include "Room.h"
#include "Meeting.h"
#include "ConnectionSettings.h"
#include "Configuration.h"
#include "CommunicationManager.h"
#include "DeviceManager.h"
#include "Clock.h"
#include "ErrorMapper.h"

#include <QtDebug>

QTime Engine::endOfTheDay = QTime( 23, 59, 0, 0 ); // end of the day is 11:59pm

Engine::Engine() :
		iClock( 0 ), iConfiguration( Configuration::instance() ), iCommunication( 0 ), iCurrentRoom( 0 )
{
	// if reading of configuration fails, signal that initialization failed
	if ( iConfiguration == 0 )
	{
		QTimer::singleShot( 0, this, SIGNAL( initializationFailed() ) );
		return;
	}
	// initialize communication
	iCommunication = new CommunicationManager( iConfiguration->connectionSettings() );
	connect( iCommunication, SIGNAL( error( int ) ), this, SLOT( errorHandler( int ) ) );
	connect( iCommunication, SIGNAL( meetingsFetched( QList<Meeting*> ) ), this, SLOT( meetingsFetched( QList<Meeting*> ) ) );

	// create application clock
	iClock = new Clock;
	connect( iClock, SIGNAL( tick( QDateTime ) ), this, SLOT( checkStatusOfAllRooms() ) );

	iAutoRefresh = new QTimer;
	iAutoRefresh->setInterval( iConfiguration->connectionSettings()->refreshInterval() * 1000 );
	iAutoRefresh->start();
	connect( iAutoRefresh, SIGNAL( timeout() ), iAutoRefresh, SLOT( start() ) );
	connect( iAutoRefresh, SIGNAL( timeout() ), this, SLOT( fetchMeetings() ) );

	qDebug() << "device";

	iDevice = new DeviceManager( iConfiguration->startupSettings() );

	QTimer::singleShot( 0, this, SLOT( fetchMeetings() ) );

	// TODO: continue implementation
}

Engine::~Engine()
{
}

Room* Engine::defaultRoom()
{
	return iConfiguration->defaultRoom();
}

Clock* Engine::clock()
{
	return iClock;
}

Configuration* Engine::configuration()
{
	return iConfiguration;
}

DeviceManager* Engine::deviceManager()
{
	return iDevice;
}

void Engine::checkStatusOfAllRooms()
{
	// iterate trough on the rooms
	for ( int i = 0; i < iConfiguration->rooms().count(); i++ )
	{
		// and check the status
		roomStatusInfoNeeded( iConfiguration->rooms().at( i ) );
	}
}

int Engine::indexOfMeetingAt( Room *aRoom, QDateTime aAt )
{
	for ( int i = 0; i < iMeetings.count(); i++ )
	{
		// exchange server ensures that there is only one meeting in a room at a specified time
		if ( aRoom->equals( iMeetings.at( i )->room() )
			  && iMeetings.at( i )->startsAt() <= aAt
			  && iMeetings.at( i )->endsAt() >= aAt )
		{
			return i;
		}
	}
	return -1;
}

int Engine::indexOfMeetingAfter( Room *aRoom, QDateTime aAfter )
{
	// seeks for the next meeting on the SAME DAY
	int min = -1;
	for ( int i = 0; i < iMeetings.count(); i++ )
	{
		// if the meeting is in the same room, on the same day but after the specified time
		if ( aRoom->equals( iMeetings.at( i )->room() )
			  && iMeetings.at( i )->startsAt().date() == aAfter.date()
			  && iMeetings.at( i )->startsAt() > aAfter )
		{
			// if there was not any meeting find yet or the previously found is a later one then the (i)th
			if ( min == -1
				  || iMeetings.at( min )->startsAt() > iMeetings.at( i )->startsAt() )
			{
				min = i;
			}
		}
	}
	return min;
}

void Engine::roomStatusInfoNeeded( Room *aRoom )
{
	if ( aRoom == 0 )
	{
		return;
	}

	int indexOfCurrentMeeting = indexOfMeetingAt( aRoom, iClock->datetime() );
	int indexOfNextMeeting = indexOfMeetingAfter( aRoom, iClock->datetime() );

//	qDebug() << QString( "Engine::roomStatusInfoNeeded\troom:%1current:%2 next:%3" ).arg( aRoom->toString() ).arg( indexOfCurrentMeeting ).arg( indexOfNextMeeting );

	// if there is no meeting, then status is Free; otherwise Busy
	Room::Status status = ( indexOfCurrentMeeting == -1 ) ? Room::FreeStatus : Room::BusyStatus;
	// if room is Busy, then check end time, otherwise...
	QTime until = ( status == Room::BusyStatus ) ? iMeetings.at( indexOfCurrentMeeting )->endsAt().time() :
			  // ...if there is meeting following on the same day then check end time, otherwise end is the of the working day
			  (( indexOfNextMeeting != -1 ) ? iMeetings.at( indexOfNextMeeting )->startsAt().time() : Engine::endOfTheDay );

	emit roomStatusChanged( aRoom, status, until );
}

void Engine::fetchMeetings()
{
	// TODO : define interval correctly. at the moment it's +/- 14 days
	Room *room = iCurrentRoom;
	if ( room == 0 ) room = defaultRoom();
	qDebug() << "Engine::fetchMeetings for " << room->name();
	fetchMeetings( iClock->datetime().addDays( -14 ), iClock->datetime().addDays( 14 ), room );
}

void Engine::fetchMeetings( const QDateTime &aFrom, const QDateTime &aUntil, Room *aIn )
{
	iCommunication->fetchMeetings( aFrom, aUntil, aIn );
}

bool Engine::isMeetingInList( QList<Meeting*> aList, Meeting *aMeeting )
{
	for ( int i = 0; i < aList.count(); i++ )
	{
		if ( aMeeting->equals( aList.at( i ) ) )
		{
			return true;
		}
	}
	return false;
}

void Engine::meetingsFetched( QList<Meeting*> aMeetings )
{
	QList<Meeting*> cpyMeetings = QList<Meeting*>( aMeetings );

	// check if there is any new meeting in the list came from the server -> added
	for ( int i = 0; i < cpyMeetings.count(); i++ )
	{
		// if the (i)th meeting is not in the local meeting list
		//if ( !iMeetings.contains( cpyMeetings.at( i ) ) )
		if ( !isMeetingInList( iMeetings, cpyMeetings.at( i ) ) )
		{
			// add to the local database =)
			iMeetings.append( cpyMeetings.at( i ) );
			// and signal the changes
			emit meetingAdded( cpyMeetings.at( i ) );
		}
	}

	// check if there is any meeting NOT in the list came from the server -> deleted
	for ( int i = 0; i < iMeetings.count(); i++ )
	{
		// if the (i)th meeting is in the local but NOT in the server's meeting list
		if ( !isMeetingInList( cpyMeetings, iMeetings.at( i ) ) )
		{
			// signal the changes
			emit meetingDeleted( iMeetings.at( i ) );
			// delete the meeting from the local list
			iMeetings.removeAt( i );
		}
	}

	// refresh room status info
	roomStatusInfoNeeded( defaultRoom() );
}

void Engine::errorHandler( int aCode )
{
	// TODO :
	// handle the error

	// inform UI about the problem
	emit error( ErrorMapper::codeToString( aCode ) );
}

void Engine::createMeeting( const Meeting *aMeeting, const QString &aUsername, const QString &aPassword )
{
	iCommunication->createMeeting( aMeeting, aUsername, aPassword );
}

void Engine::updateMeeting( const Meeting *aMeeting, const QString &aUsername, const QString &aPassword )
{
	iCommunication->updateMeeting( aMeeting, aUsername, aPassword );
}

void Engine::deleteMeeting( const Meeting *aMeeting, const QString &aUsername, const QString &aPassword )
{
	iCommunication->deleteMeeting( aMeeting, aUsername, aPassword );
}

void Engine::currentRoomChanged( Room *aCurrentRoom )
{
	iCurrentRoom = aCurrentRoom;
	qDebug() << "Engine::currentRoomChanged to " << iCurrentRoom->name();
}
