/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "speeddisplay.h"

#include <QFontMetrics>
#include <QLocale>
#include <QPainter>
#include <QRectF>

//! Constructor
SpeedDisplay::SpeedDisplay(int width, int height, QGraphicsItem * parent):
        DisplayBox(width, height, parent),
        speed("0"),
        speedType(KMTYPE),
        defaultBackgroundColor(0, 100, 255, 100),
        speedKilometers(0),
        mileConvertValue(1.609344)
{
    speedTypeText[KMTYPE].append(" km/h") ;
    speedTypeText[MILETYPE].append(" mph") ;

    displayText.append(speed) ;
    displayText.append(speedTypeText[speedType]) ;

    this->height = qMin(height, width) ;
    this->width = qMax(width, height) ;
    QLocale locale ;
    speedType = locale.measurementSystem() ;

    measure() ;
    createDisplayValue() ;
}

//! destructor
SpeedDisplay::~SpeedDisplay()
{
}

//! Used to measure text drawing area and measure fonts size.
void SpeedDisplay::measure()
{
    speedTextArea.setX(this->boundingRect().x()+2) ;
    speedTextArea.setTop(this->boundingRect().top()+2) ;
    speedTextArea.setBottom(this->boundingRect().bottom()-2) ;
    speedTextArea.setRight(this->boundingRect().right()-2) ;

    speedFont.setFamily("sans-serif") ;
    speedFont.setBold(true) ;

    //! First set pixel size by giving height of the area.
    speedFont.setPixelSize(static_cast<int> (this->speedTextArea.height()));

    //! Calculates relative size for the text width.
    speedFont.setStretch(static_cast<int> (((this->speedTextArea.width()/8)/this->speedTextArea.height())*100));
}

//! converts kilometers to miles.
int SpeedDisplay::calculateMileValue()
{
    int speedMile = 0 ;
    speedMile = static_cast<int> (speedKilometers/mileConvertValue);
    return speedMile ;
}

//! Public function for speed update.
void SpeedDisplay::updateSpeed(qreal speed)
{
    this->speedKilometers = (int)speed ;
    createDisplayValue() ;
}

//! creates new display value based on system metrics and given speed value.
void SpeedDisplay::createDisplayValue()
{
    this->speed.clear();
    if(speedType == KMTYPE)
    {        
        this->speed.setNum(speedKilometers) ;
    }
    else if(speedType == MILETYPE)
    {
        this->speed.setNum(calculateMileValue()) ;
    }

    displayText.clear();
    displayText.append(speed) ;
    displayText.append(speedTypeText[speedType]) ;
}

//! rect what bounds the item.
QRectF SpeedDisplay::boundingRect() const
{
    return QRectF(0, 0, width, height);
}

//! paint function where graphics are painted.
void SpeedDisplay::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option) ;
    Q_UNUSED(widget) ;

    DisplayBox::paint(painter, option, widget);

    int height = DisplayBox::getHeight();
    int width = DisplayBox::getWidth();

    //! Draw rounding rectangle.
    painter->drawRect(static_cast<int> (boundingRect().x()),
                      static_cast<int> (boundingRect().y()),
                      width, height);

    //! Draw text to screen.
    painter->setPen(Qt::black) ;
    painter->setFont(speedFont) ;
    painter->drawText(boundingRect(), Qt::AlignCenter, displayText, &speedTextArea) ;
}
