/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __SERIALPORT_H__
#define __SERIALPORT_H__

#include <QIODevice>
#include <termios.h>

class QFile;
class QSocketNotifier;

/*!
 * Class SerialPort provides the functionality to read from
 * and write to a serial port in a UNIX system.
 */
class SerialPort : public QIODevice
{
    Q_OBJECT

public:
    /*!
     * \brief Constructor.
     *
     * Constructs a SerialPort object using the default serial device.
     * The default is /dev/ttyS0.
     */
    SerialPort();

    /*!
     * \brief Constructor.
     *
     * Constructs a SerialPort object using a custom serial device.
     *
     * \param portName The name of the serial port device file.
     */
    SerialPort(QString portName);

    //! Destructor.
    ~SerialPort();

    /*!
     * Open the serial port.
     *
     * \param mode The device open mode as described in QIODevice::OpenMode.
     * \return True if port was opened successfully, false otherwise.
     */
    bool open(OpenMode mode);

    //! Close the serial port.
    void close();

    /*!
     * \brief Get available bytes.
     *
     * \return The number of bytes available for reading from the serial port.
     */
    virtual qint64 bytesAvailable() const;

private:
    //! The default serial port device file name.
    const QString deviceFileName;

    //! The serial port device file.
    QFile *deviceFile;

    //! Monitors serial port activity.
    QSocketNotifier *notifier;

    //! Used to store the original serial port settings and restore them later.
    struct termios oldSettings;

private slots:
    /*!
     * \brief Notify of available data.
     *
     * This slot function is connected to the activated() signal of
     * the notifier member, which is received when there is data waiting
     * to be read from the serial port. The function just emits the
     * readyRead() signal and does nothing else.
     */
    void notifierActivated();

protected:
    /*!
     * \brief Read data from serial port.
     *
     * Reads data from the serial port. This function is called by
     * QIODevice when using the class' data reading functions.
     *
     * \param data A buffer where the read data will be stored.
     * \param maxSize Maximum number of bytes to read.
     * \return The number of bytes read.
     */
    virtual qint64 readData(char *data, qint64 maxSize);

    /*!
    * \brief Write data to serial port.
    *
    * Writes data to the serial port. This function is called by
    * QIODevice when using the class' data writing functions.
    *
    * \param data A buffer storing the data to write to the serial port.
    * \param maxSize Maximum number of bytes to write.
    * \return The number of bytes written.
    */
    virtual qint64 writeData(const char *data, qint64 maxSize);
};

#endif // __SERIALPORT_H__
