/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifdef DEBUG
#include <QDebug>
#endif

#include <QFile>
#include <QSocketNotifier>
#include <QString>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include "serialport.h"

//! Constructor.
SerialPort::SerialPort() :
        deviceFileName("/dev/ttyS0")
{
    deviceFile = new QFile(deviceFileName);
    notifier = NULL;
}

//! Constructor.
SerialPort::SerialPort(QString portName)
{
    deviceFile = new QFile(portName);
    notifier = NULL;
}

//! Destructor.
SerialPort::~SerialPort()
{
    if (isOpen()) {
        close();
    }

    deviceFile->close();
    delete deviceFile;

    if (notifier != NULL) {
        delete notifier;
    }
}

//! Open the serial port.
bool SerialPort::open(OpenMode mode)
{
    const int error = -1;

    // A structure for setting the serial port settings.
    struct termios newSettings;

    // If the port is already open, do nothing.
    if (!isOpen()) {
        // Open the serial port device file.
        if (!deviceFile->open(QIODevice::ReadWrite)) {
#ifdef DEBUG
            qDebug() << "Error: Failed to open device file.";
#endif
            return false;
        }

        QIODevice::open(mode);

        /*
         * Create the notifier, attach it to the serial port device
         * and tell it to monitor the port for readable data.
         */
        notifier = new QSocketNotifier(deviceFile->handle(),
                                       QSocketNotifier::Read);
        connect(notifier, SIGNAL(activated(int)),
                this, SLOT(notifierActivated()));

        // Store the current serial port settings so they can be restored later.
        if (tcgetattr(deviceFile->handle(), &oldSettings) == error) {
#ifdef DEBUG
            qDebug() << "Error: tcgetattr() failed:" << strerror(errno);
#endif
            return false;
        }

        // Clear the structure before setting the values.
        bzero(&newSettings, sizeof(newSettings));

        // Set the serial port speed.
        cfsetispeed(&newSettings, B115200);
        cfsetospeed(&newSettings, B115200);

        // Set the serial port flags.
        newSettings.c_cflag = B115200 | CS8 | CLOCAL | CREAD;
        newSettings.c_iflag = 0;
        newSettings.c_oflag = 0;
        newSettings.c_lflag = ICANON;
        newSettings.c_cc[VMIN] = 0;
        newSettings.c_cc[VTIME] = 0;
        newSettings.c_cc[VSUSP] = 0;

        // Apply the new settings.
        if (tcsetattr(deviceFile->handle(), TCSANOW, &newSettings) != 0) {
#ifdef DEBUG
            qDebug() << "Error: tcsetattr() failed:" << strerror(errno);
#endif
            return false;
        }

        return true;
    }

    return false;
}

//! Close the serial port.
void SerialPort::close()
{
    const int error = -1;

    if (isOpen()) {
        // Restore the original serial port settings.
        if (tcsetattr(deviceFile->handle(), TCSANOW, &oldSettings) == error) {
#ifdef DEBUG
            qDebug() << "Error: tcsetattr() failed:" << strerror(errno);
#endif
        }

        if (notifier != NULL) {
            delete notifier;
            notifier = NULL;
        }

        deviceFile->close();
        QIODevice::close();
    }
}

//! Read data from serial port.
qint64 SerialPort::readData(char *data, qint64 maxSize)
{
    const int error = -1;
    qint64 bytesRead;

    if (isOpen()) {
        if ((bytesRead = deviceFile->read(data, maxSize)) == error) {
#ifdef DEBUG
            qDebug() << "Error: failed to read data from file.";
#endif
        }

        return bytesRead;
    }

    return error;
}

//! Write data to serial port.
qint64 SerialPort::writeData(const char *data, qint64 maxSize)
{
    const int error = -1;
    qint64 bytesWritten;

    if (isOpen()) {
        if ((bytesWritten = deviceFile->write(data, maxSize)) == error) {
#ifdef DEBUG
            qDebug() << "Error: failed to write data to file.";
#endif
        }

        // Write the file buffer to disk.
        deviceFile->flush();

        return bytesWritten;
    }

    return error;
}

//! Get available bytes.
qint64 SerialPort::bytesAvailable() const
{
    const int error = -1;
    int n;
    fd_set set;
    struct timeval tv;
    int bytesAvailable;

    if (isOpen()) {
        FD_ZERO(&set);
        FD_SET(deviceFile->handle(), &set);

        tv.tv_sec = 0;
        tv.tv_usec = 0;

        // Check if there's new data waiting to be read from the port.
        if ((n = select(deviceFile->handle() + 1, &set, NULL, &set, &tv))
                == error) {
#ifdef DEBUG
            qDebug() << "Error: select() failed." << strerror(errno);
#endif

            return error;
        }

        // Find out how many bytes are available.
        if (ioctl(deviceFile->handle(), FIONREAD, &bytesAvailable) == error) {
#ifdef DEBUG
            qDebug() << "Error: ioctl() failed." << strerror(errno);
#endif

            return error;
        }

        // Return the number of bytes available.
        return bytesAvailable + QIODevice::bytesAvailable();
    }

    // No new data available.
    return 0;
}

//! Notify of available data.
void SerialPort::notifierActivated()
{
    emit readyRead();
}
