/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "gpssatellitedata.h"
#include "gpsdata.h"
#include "point.h"
#include "satellitedetailsdialog.h"
#include <QObject>
#include <QDebug>
#include <QTime>
#include <QVBoxLayout>
#include "diagrams.h"
#include "definitions.h"

//! constructor
SatelliteDetailsDialog::SatelliteDetailsDialog(GpsData &gpsData,
                                               Point &point,
                                               QWidget *parent) :
        QDialog(parent),
        degreeSign("°"),
        kilometersPerHour(" km/h"),
        meters(" m"),
        timeFormat("hh:mm:ss")
{
    setupUi(this);
    diag = NULL;
    QString degree(degreeSign);
    degree.remove(0, 1);

    fixQualityText[0] = "Invalid";
    fixQualityText[1] = "SPS";
    fixQualityText[2] = "DGPS";
    fixQualityText[3] = "PPS";
    fixQualityText[4] = "RTK";
    fixQualityText[5] = "FRTK";
    fixQualityText[6] = "Estim";
    fixQualityText[7] = "Manual";
    fixQualityText[8] = "Simulation";

    fix3d[0] = "no fix";
    fix3d[1] = "2D";
    fix3d[2] = "3D";

    QString latitude;
    latitude.setNum(gpsData.getLatitude());
    latitude.append(degree);
    latitudeValueLabel->setText(latitude);

    QString longitude;
    longitude.setNum(gpsData.getLongitude());
    longitude.append(degree);
    longitudeValueLabel->setText(longitude);

    QString speed;
    speed.setNum(gpsData.getSpeed());
    speed.append(kilometersPerHour);
    speedValueLabel->setText(speed);

    QString altitude;
    altitude.setNum(point.getAltitude());
    altitude.append(meters);
    altitudeValueLabel->setText(altitude);

    QString heading;
    heading.setNum(gpsData.getHeading());
    heading.append(degree);
    headingValueLabel->setText(heading);

    QTime *time = gpsData.getSystemTime();

    QString timeString = time->toString(timeFormat);
    timeValueLabel->setText(timeString);

    QString satellitesInView;
    satellitesInView.setNum(gpsData.getSatellitesInView());
    satellitesInViewValueLabel->setText(satellitesInView);

    QString satellitesInUse;
    satellitesInUse.setNum(gpsData.getSatellitesInUse());
    satellitesInUseValueLabel->setText(satellitesInUse);

    fixValueLabel->setText(fix3d[gpsData.getFix() - 1]);

    fixQualityValueLabel->setText(fixQualityText[gpsData.getFixQuality()]);

    QString maxSpeed;
    maxSpeed.setNum(gpsData.getMaxSpeed());
    maxSpeed.append(kilometersPerHour);
    maxSpeedValueLabel->setText(maxSpeed);

    connect(&gpsData, SIGNAL(valueChanged(const QString &, qreal)),
            this, SLOT(changeValue(const QString &, qreal)));

    connect(&gpsData, SIGNAL(timeChanged(QTime &)),
            this, SLOT(changeTime(QTime &)));

    connect(tabWidget, SIGNAL(currentChanged(int)),
            this, SLOT(tabChanged(int)));
}

//! destructor
SatelliteDetailsDialog::~SatelliteDetailsDialog()
{
}

//! when user changes tabs, construct or destruct the SNR bars
void SatelliteDetailsDialog::tabChanged(int index)
{
    const int diagramMinValue = 1;
    const int diagramMaxValue = 100;

    if (index == 1) {
        diag = new Diagrams(maxSatellitesCount);
        diag->setMinimumValue(diagramMinValue);
        diag->setMaximumValue(diagramMaxValue);

        for (int i = 0; i < maxSatellitesCount; i++) {
            connect(&gpsSatelliteDataArray[i], SIGNAL(snrChanged(int, int)),
                    this, SLOT(updateBars(int, int)));

            diag->setValue(i, gpsSatelliteDataArray[i].getSnr());

            QString id;
            id.setNum(gpsSatelliteDataArray[i].getPrn());

            if (gpsSatelliteDataArray[i].getPrn() == 0) {
                id.clear();
                id.append(" ");
            }
            else if (id.length() < 2) {
                id.insert(0, "0");
            }

            diag->setId(i, id);
            diag->setElevation(i, gpsSatelliteDataArray[i].getElevation());
            diag->setAzimuth(i, gpsSatelliteDataArray[i].getAzimuth());
        }

        vbox = new QVBoxLayout;
        vbox->addWidget(diag);
        tabWidget->currentWidget()->setLayout(vbox);
        diag->repaint();
    }
    else if (diag != NULL) {
        for (int i = 0; i < diag->getCountOfBars(); i++) {
            QObject::disconnect(&gpsSatelliteDataArray[i], 0,
                                this, SLOT(updateBars(int, int)));
        }

        delete diag;
        diag = NULL;
        delete vbox;
    }
}

//! for updating the bars that visualize satellites' SNR
void SatelliteDetailsDialog::updateBars(int prn, int snr)
{
    for (int i = 0; i < maxSatellitesCount; i++) {
        if (gpsSatelliteDataArray[i].getPrn() != 0) {
            QString id;
            id.setNum(prn);

            if (id.length() < 2) {
                id.insert(0, "0");
            }

            if (id.compare(diag->getId(i)) == 0) {
                diag->setAzimuth(i, gpsSatelliteDataArray[i].getAzimuth());
                diag->setElevation(i, gpsSatelliteDataArray[i].getElevation());
                diag->setValue(i, snr);
                diag->repaint();
                break;
            }
        }
    }
}

//! for changing the displayed values other than time
void SatelliteDetailsDialog::changeValue(const QString &identifier, qreal value)
{
    const QString latitudeIdentifier("latitude");
    const QString longitudeIdentifier("longitude");
    const QString speedIdentifier("speed");
    const QString altitudeIdentifier("altitude");
    const QString headingIdentifier("heading");
    const QString satellitesInViewIdentifier("satellitesInView");
    const QString satellitesInUseIdentifier("satellitesInUse");
    const QString fixValueIdentifier("fixValue");
    const QString fixQualityIdentifier("fixquality");
    const QString maxSpeedIdentifier("maxspeed");

    QString degree(degreeSign);
    degree.remove(0, 1);

    if (identifier.compare(latitudeIdentifier) == 0) {
        QString latitude;
        latitude.setNum(value);
        latitude.append(degree);
        latitudeValueLabel->setText(latitude);
    }
    else if (identifier.compare(longitudeIdentifier) == 0) {
        QString longitude;
        longitude.setNum(value);
        longitude.append(degree);
        longitudeValueLabel->setText(longitude);
    }
    else if (identifier.compare(speedIdentifier) == 0) {
        QString speed;
        speed.setNum(value);
        speed.append(kilometersPerHour);
        speedValueLabel->setText(speed);
    }
    else if (identifier.compare(altitudeIdentifier) == 0) {
        QString altitude;
        altitude.setNum(value);
        altitude.append(meters);
        altitudeValueLabel->setText(altitude);
    }
    else if (identifier.compare(headingIdentifier) == 0) {
        QString heading;
        heading.setNum(value);
        heading.append(degree);
        headingValueLabel->setText(heading);
    }
    else if (identifier.compare(satellitesInViewIdentifier) == 0) {
        QString satellitesInView;
        satellitesInView.setNum(value);
        satellitesInViewValueLabel->setText(satellitesInView);
    }
    else if (identifier.compare(satellitesInUseIdentifier) == 0) {
        QString satellitesInUse;
        satellitesInUse.setNum(value);
        satellitesInUseValueLabel->setText(satellitesInUse);
    }
    else if (identifier.compare(fixValueIdentifier) == 0) {
        fixValueLabel->setText(fix3d[(int)value - 1]);
    }
    else if (identifier.compare(fixQualityIdentifier) == 0) {
        fixQualityValueLabel->setText(fixQualityText[(int)value]);
    }
    else if (identifier.compare(maxSpeedIdentifier) == 0) {
        QString maxSpeed;
        maxSpeed.setNum(value);
        maxSpeed.append(kilometersPerHour);
        maxSpeedValueLabel->setText(maxSpeed);
    }
}

//! for changing the displayed time
void SatelliteDetailsDialog::changeTime(QTime &systemTime)
{
    QString timeString = systemTime.toString(timeFormat);
    timeValueLabel->setText(timeString);
}
