/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QPainter>
#include "satelliteDrawing.h"
#include "unitconverter.h"
#include "math.h"

//! Constructor, parameters as ints
SatelliteDrawing::SatelliteDrawing(int x, int y, int width, int height) :
        x(x),
        y(y),
        width(width),
        height(height)
{
    init();
}

//! Constructor, parameters inside a QRect
SatelliteDrawing::SatelliteDrawing(QRect &rect)
{
    this->x = rect.x();
    this->y = rect.y();
    this->width = rect.width();
    this->height = rect.height();
    init() ;
}

//! Destructor
SatelliteDrawing::~SatelliteDrawing()
{
}

/*!
 * This function initializes all variables and is called from both constructors.
 */
void SatelliteDrawing::init()
{
    idPen = QPen(Qt::black);
    elevation = 0;
    azimuth = 0;
    modelStyle = QBrush(Qt::gray);
}

//! Sets new Id for the satellite.
void SatelliteDrawing::setId(QString &id)
{
    this->id = id;
}

//! Sets elevation value for the satellite
void SatelliteDrawing::setElevation(int elevation)
{
    this->elevation = elevation;
}

//! Sets azimuth value for the satellite
void SatelliteDrawing::setAzimuth(int azimuth)
{
    this->azimuth = azimuth;
}

//! Updates drawing area
void SatelliteDrawing::updateArea(QRect &rect)
{
    this->x = rect.x();
    this->y = rect.y();
    this->width = rect.width();
    this->height = rect.height();
}

/*!
 * Calculates satellites position on the plot by the given azimuth and
 * elevation values.
 */
void SatelliteDrawing::calculatePosition()
{
    const int modelRectSize = 10;

    UnitConverter *uc = new UnitConverter();
    int tmp = (int)(uc->deg2rad(azimuth));
    modelRect.setX((int)(x + width / 2 +
                   (90 - elevation) * (width / 2) / 90 * sinf(tmp)));
    modelRect.setY((int)(y + height / 2 -
                   (90 - elevation) * (height / 2) / 90 * cosf(tmp)));
    modelRect.setX(modelRect.x());
    modelRect.setY(modelRect.y());
    modelRect.setHeight(modelRectSize);
    modelRect.setWidth(modelRectSize);
    delete uc;
}

//! Called by paintevent. Draws the circle describing the satellite and it's ID
void SatelliteDrawing::update(QPainter *painter)
{
    if (azimuth >= 0 && elevation >= 0 && id.length() > 1) {
        this->calculatePosition();
        painter->setPen(Qt::gray);
        painter->setBrush(modelStyle);
        painter->drawEllipse(modelRect);
        idPen.setColor(Qt::black);
        painter->setPen(idPen);
        font.setPointSize((int)((modelRect.height() / 2) * sqrt(2)));
        painter->setFont(font);
        painter->drawText(modelRect, id);
    }
}
