/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QObject>
#include <QRegExp>
#include <QUrl>
#include <QWebFrame>
#include "routedownloader.h"
#include "routedata.h"

//! Constructor.
RouteDownloader::RouteDownloader(QObject *routeData, QObject *parent) :
    GoogleMapsDownloader(parent),
    downloadAddress("http://qtmapper.garage.maemo.org/routedownload.html")
{
    this->routeData = qobject_cast<RouteData *> (routeData);

    connect(routeData, SIGNAL(routeDataReady()),
            this, SLOT(readDataFromJavaScriptObject()));
    connect(routeData, SIGNAL(routeDownloadError()),
            this, SLOT(downloadFailed()));
}

//! Download route data.
void RouteDownloader::downloadData(const QString &fromAddress,
                                   const QString &toAddress,
                                   const QString &locale)
{
    this->fromAddress = fromAddress;
    this->toAddress = toAddress;

    QString urlString(downloadAddress);
    urlString.append("?from=");
    urlString.append(fromAddress);
    urlString.append("&to=");
    urlString.append(toAddress);
    urlString.append("&locale=");
    urlString.append(locale);
    
    emit readyToDownload(urlString);
}

//! Reads the route data from the GDirections object in the Javascript code.
void RouteDownloader::readDataFromJavaScriptObject()
{
    const QString emptyString("");
    const QString htmlTagReplaceRegExp("<[^>]*>");
    const QString nonBreakingSpace("&nbsp;");
    const QString spaceCharacter(" ");
    const QString addressSeparator(" - ");
    const QString script(
    "routeData.setTotalDistance(gdir.getDistance().html);"
    "for (i = 0; i < gdir.getNumRoutes(); ++i) {"
    "    for (j = 0; j < gdir.getRoute(i).getNumSteps(); ++j) {"
    "        routeData.addWaypoint(gdir.getRoute(i)."
    "                              getStep(j).getDescriptionHtml(),"
    "                              gdir.getRoute(i)."
    "                              getStep(j).getLatLng().lng(),"
    "                              gdir.getRoute(i)."
    "                              getStep(j).getLatLng().lat())"
    "    }"
    "}"
    "for (i = 0; i < gdir.getPolyline().getVertexCount(); ++i) {"
    "    routeData.addLinePoint(gdir.getPolyline().getVertex(i).lng(),"
    "                           gdir.getPolyline().getVertex(i).lat())"
    "}"
    );

    mainFrame()->evaluateJavaScript(script);

    // Strip HTML codes from the routeData strings.
    routeData->setTotalDistance(routeData->getTotalDistance()
                                .replace(QRegExp(htmlTagReplaceRegExp),
                                         emptyString)
                                .replace(nonBreakingSpace, spaceCharacter));

    for (int i = 0; i < routeData->getWaypoints().size(); i++) {
        Waypoint *wp = routeData->getWaypoints().at(i);
        wp->setName(wp->getName().replace(QRegExp(htmlTagReplaceRegExp),
                                          emptyString));
    }

    QString routeName(fromAddress);
    routeName.append(addressSeparator);
    routeName.append(toAddress);
    routeData->setName(routeName);

    emit completed();
}

//! If the GDirections object reports an error, this is called.
void RouteDownloader::downloadFailed()
{
    emit failed();
}
