/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __ROUTEDATA_H__
#define __ROUTEDATA_H__

#include <QObject>
#include <QString>
#include <QVector>
#include "linepoint.h"
#include "waypoint.h"

//! Class contains route information.
class RouteData : public QObject
{
    Q_OBJECT

public:
    //! Constructor.
    RouteData(QObject *parent = 0);

    //! Destructor.
    ~RouteData();
    
    /*!
     * Adds a new waypoint in the waypoints vector.
     *
     * \param waypoint The waypoint to add.
     */
    void addWaypoint(Waypoint *waypoint);

    /*!
     * Removes a waypoint from the waypoints vector.
     *
     * \param index The index of the waypoint to remove.
     */
    void removeWaypoint(int index);

    /*!
     * Adds a new line point in the linePoints vector.
     *
     * \param linePoint The line point to add.
     */
    void addLinePoint(LinePoint *linePoint);

    /*!
     * Removes a line point from the linePoints vector.
     *
     * \param index The index of the line point to remove.
     */
    void removeLinePoint(int index);

    /*!
     * Get the total distance of the route.
     *
     * \return The total distance of the route.
     */
    QString getTotalDistance() const { return totalDistance; }

    //! Clears the route information.
    void clear();
    
public slots:
    //! Returns the name property.
    QString getName() const { return name; }

    //! Sets the name property.
    void setName(const QString &name) { this->name = name; }

    //! Returns the waypoints property.
    QVector<Waypoint *> getWaypoints() { return waypoints; }

    //! Returns the linePoints property.
    QVector<LinePoint *> getLinePoints() { return linePoints; }

    /*!
     * Adds a new waypoint in the waypoints vector.
     *
     * \param name The name of the waypoint.
     * \param longitude Longitude at the waypoint.
     * \param latitude Latitude at the waypoint.
     * \heading Heading at the waypoint.
     */
    void addWaypoint(QString name, qreal longitude, qreal latitude);

    /*!
     * Adds a new line point in the linePoints vector.
     *
     * \param longitude Longitude at the line point.
     * \param latitude Latitude at the line point.
     */
    void addLinePoint(qreal longitude, qreal latitude);

    /*!
     * Sets the total distance of the route.
     *
     * \param totalDistance The distance to set.
     */
    void setTotalDistance(const QString &totalDistance) { this->totalDistance =
                                                          totalDistance; }

    //! This is called from the JavaScript code when the route is downloaded.
    void routeDownloadCompleted();

    //! This is called from the JavaScript code if the route download failed.
    void routeDownloadFailed();
    
private:
    //! Name of this object.
    const QString objectName;

    //! The name of the route.
    QString name;

    //! A list of route points.
    QVector<Waypoint *> waypoints;

    //! A vector of points to draw a line along the route.
    QVector<LinePoint *> linePoints;

    //! The total distance of the route.
    QString totalDistance;

signals:
    //! Emitted when the route download is complete.
    void routeDataReady();

    //! Emitted when there was an error downloading the route.
    void routeDownloadError();
};

#endif // __ROUTEDATA_H__
