/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __QUICKMENUBAR_H__
#define __QUICKMENUBAR_H__

#include <QGraphicsItem>
#include <QObject>

class QColor;
class QuickMenuButton;
class QGraphicsSceneMouseEvent;
class QPainter;
class QStyleOptionGraphicsItem;
class QWidget;

//! Quickmenubar class
class QuickMenubar : public QObject, public QGraphicsItem
{
    Q_OBJECT

public:
    /*!
     * Constructor.
     * \param parent QGraphics parent.
     * \param x x position.
     * \param y y position.
     * \param width width of the bar.
     * \param backgroundColor color for the quickmenubar.
     * Default value is QColor(0, 100, 255, 100).
     */
    QuickMenubar(QGraphicsItem *parent = 0, int x = 0, int y = 0,
                 int width = 50, QColor *backgroundColor = NULL);

    /*!
     * Destructor.
     */
    ~QuickMenubar();

    /*!
     * Bounding rect is have to implement.
     */
    QRectF boundingRect() const;

    /*!
     * paint event need this.
     * \param painter QPainter object.
     * \param option not in use.
     * \param widget not in use.
     */
    void paint(QPainter *painter,
               const QStyleOptionGraphicsItem *option,
               QWidget *widget = 0);

    /*!
     * Sets background color for the quickmenu.
     * Default value is QColor(0, 100, 255, 100).
     */
    void setBackgroundColor(const QColor &);

    /*!
     * Set new x and y positions for the Quickmenu
     * \param x x position
     * \param y y position
     */
    void setPos(int x, int y);

    /*!
     * Sets tracking button to "ON-mode".
     * If users selects tracking on from the drop-down menu this
     * is called to indicate tracking ON.
     * \param bool where true means tracking on and false tracking is off.
     */
    void setTrackingSelected(bool);

    qreal getWidth() const { return this->width; }

public slots:
    /*!
     * This function must be implemented when mouse action wants to catch.
     * \param e Information about the happened event.
     */
    void mousePressed(QGraphicsSceneMouseEvent *e);

    /*!
     * This function must be implemented when mouse action wants to catch.
     * \param e Information about the happened event.
     */
    void mouseReleased(QGraphicsSceneMouseEvent *e);

signals:
    /*!
     * When user pressing out side of the quickmenu this signal
     * function is emitted.
     * @param x is mouses' global x-position.
     * @param y is mouses' global y-position.
     */
    void poiSelected(int x, int y);

    /*!
     * When user pressing "plus" -sign button this function will be emitted.
     */
    void zoomIn();

    /*!
     * When user pressing "minus" -sign button this function will be emitted.
     */
    void zoomOut();

    /*!
     * When user pressing tracking button this function is emitted and
     * state is true or false.
     * \param state Tracking state.
     */
    void tracking(bool state);

private:
    /*!
     * Mouse coordinates is needed when Poi is selected by clicking the screen.
     * This variable is needed because signal emitted in release function
     * and coordinates is needed for the parameter.
     */
    QPoint poiPoints;

    /*!
     * Indicates when poi selection signal allowed to emit.
     */
    bool enablePoiEmit;

    /*!
     * This function is for the measuring purpose. Calculates positions
     * for the buttons and graphics.
     */
    void measure();

    /*!
     * This function measures size for the buttons.
     * Buttons max size is the given width minus ten percent.
     */
    void measureButtons();

    /*!
     * Inits all needed variables and creates objects.
     * Called from constructor only.
     */
    void init();

    //! Enum type for the button identification use.
    typedef enum { buttonZoomIn = 0, buttonZoomOut, buttonPoi, buttonTrack };

    //! Default background color.
    const QColor defaultBackgroundColor;

    //! How many buttons is displayed on the screen.
    const int buttonCount;

    //! QString type Object-array for the icons filename with paths.
    QString *iconFileName[4];

    //! GuiButton type object-array for the buttons.
    QuickMenuButton *button[4];

    //! Object for the background color.
    QColor *backgroundColor;

    //! keeps track what was the last pressed button. Allows toggled buttons.
    int lastPressedButton;

    //! x position on for Quickmenubar which is related to scene
    qreal x;

    //! y position on for Quickmenubar which is related to scene
    qreal y;

    //! width of the Quickmenubar
    qreal width;

    //! height of the Quickmenubar
    qreal height;

    /*! Length describes calculated side length.
     * Length is smaller value of button height or quickmenubar width.*/
    qreal length;
};

#endif // __QUICKMENUBAR_H__
