/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "quickmenubar.h"
#include "quickmenubutton.h"
#include "definitions.h"
#include <QtGui>
#include <QColor>
#include <QRectF>

//! Constructor.
QuickMenubar::QuickMenubar(QGraphicsItem *parent, int x, int y,
                           int width, QColor *backgroundColor) :
        QGraphicsItem(parent),
        defaultBackgroundColor(0, 100, 255, 100),
        buttonCount(4)
{
    this->width = width;
    this->height = this->width * buttonCount;
    this->x = x;
    this->y = y;

    if (backgroundColor == NULL) {
        this->backgroundColor = new QColor(defaultBackgroundColor);
    }
    else {
        this->backgroundColor = new QColor(*backgroundColor);
    }

    init();
    measure();
}

//! Destructor.
QuickMenubar::~QuickMenubar()
{
    for (int i = 0; i < 4; i++) {
        delete iconFileName[i];
    }

    for (int i = 0; i < buttonCount; i++) {
        delete button[i];
    }

    delete backgroundColor;
}

//! Set new x and y positions for the Quickmenu
void QuickMenubar::setPos(int x, int y)
{
    this->x = x;
    this->y = y;
}

//! Bounding rect is have to implement.
QRectF QuickMenubar::boundingRect() const
{
    return QRectF(this->x, this->y, this->width, this->height);
}

//! paint event need this.
void QuickMenubar::paint(QPainter *painter,
                         const QStyleOptionGraphicsItem *option,
                         QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    const QColor lineColor(0, 100, 255, 50);
    measure();

    QBrush bgBrush(*backgroundColor);
    painter->setBrush(bgBrush);
    painter->setPen(lineColor);

    QRectF rectTemp(x, y, width, height) ;

    painter->drawRect(rectTemp);

    for (int i = 0; i < buttonCount; i++) {
        button[i]->update(painter);
    }
}

// catch pressed button and set it selected -mode.
void QuickMenubar::mousePressed(QGraphicsSceneMouseEvent* e)
{
    bool poiChanged = false;
    int mouseXpos = (int)(e->scenePos().x());
    int mouseYpos = (int)(e->scenePos().y());

    //create QRect for the comparison purpose.
    QRectF area(this->x, this->y, this->width, this->height);

    //first check if last selection was poi.
    if (this->lastPressedButton == buttonPoi) {
        if (!area.contains(mouseXpos, mouseYpos)) {
            poiChanged = true;
            enablePoiEmit = true;
            poiPoints.setX((int)mouseXpos);
            poiPoints.setY((int)mouseYpos);
            button[this->lastPressedButton]->setMode(QIcon::Normal);
            this->lastPressedButton = -1;
        }
        else {
            /* Last selection was poi button and now the user clicks the
             * quickmenu again. We simply de-select the poi button. */
            button[this->lastPressedButton]->setMode(QIcon::Normal);
            this->lastPressedButton = -1;
            return;
        }
    }

    if (button[buttonPoi]->getArea().contains(mouseXpos, mouseYpos) &&
        poiChanged == false) {
        // check was the pressed button POI selection.
        this->lastPressedButton = buttonPoi;

        if (button[buttonPoi]->getMode() == QIcon::Selected) {
            button[buttonPoi]->setMode(QIcon::Normal);
        }
        else {
            button[buttonPoi]->setMode(QIcon::Selected);
        }
    }// if buttonPoi
    else if (button[buttonZoomOut]->getArea().contains(mouseXpos, mouseYpos)) {
        this->lastPressedButton = buttonZoomOut;
        button[buttonZoomOut]->setMode(QIcon::Selected);
        this->update();
    }// if buttonZoomOut
    else if (button[buttonZoomIn]->getArea().contains(mouseXpos, mouseYpos)) {
        this->lastPressedButton = buttonZoomIn;
        button[buttonZoomIn]->setMode(QIcon::Selected);
        this->update();
    }// if buttonZoomIn
    else if (button[buttonTrack]->getArea().contains(mouseXpos, mouseYpos)) {
        this->lastPressedButton = buttonTrack;
        /* if track button is selected, check its state and change it
         * Emits the signal "tracking"*/
        if (button[buttonTrack]->getMode() == QIcon::Selected) {
            button[buttonTrack]->setMode(QIcon::Normal);
            emit tracking(false);
        }
        /* else if the mode wasn't selected change it and emit
         * tracking off signal.*/
        else {
            button[buttonTrack]->setMode(QIcon::Selected);
            emit tracking(true);
            /* if buttonPoi was toggled it will changed to normal state.
             * No pois here.*/
            if (button[buttonPoi]->getMode() == QIcon::Selected) {
                button[buttonPoi]->setMode(QIcon::Normal);
            }
        }
    }
}

// release the last pressed button.
void QuickMenubar::mouseReleased(QGraphicsSceneMouseEvent* e)
{
    if (enablePoiEmit == true) {
        this->update();
        enablePoiEmit = false;
        emit poiSelected(poiPoints.x(), poiPoints.y());
    }
    else if (this->lastPressedButton <= 1 && this->lastPressedButton >= 0) {
        button[this->lastPressedButton]->setMode(QIcon::Normal);
        if (lastPressedButton == buttonZoomIn &&
            button[buttonZoomIn]->getArea()
            .contains((int)(e->scenePos().x()), (int)(e->scenePos().y()))) {
            this->update();
            emit zoomIn();
        }
        else if (lastPressedButton == buttonZoomOut &&
                 button[buttonZoomOut]->getArea()
                   .contains((int)(e->scenePos().x()), (int)(e->scenePos().y()))) {
            this->update();
            emit zoomOut();
        }
    }
    else {
        this->update();
    }
}

//! This function measures size for the buttons.
void QuickMenubar::measureButtons()
{
    //this measures same value for the buttons height and width.
    qreal h = this->height / 4;
    qreal w = this->width;

    length = qMin(h,w) ;

    //narrow button size 20% smaller.
    length *= 0.8;
}

//! This function is for the measuring purpose
void QuickMenubar::measure()
{
    QRect rect;
    this->measureButtons();
    this->height = buttonCount * this->width;
    qreal space = length * 0.1;

    //calculate iters value 2 buttons upward to find starting y -position.
    qreal yPosIter = this->y + this->height / 2 - (1.5 * space) - (2 * length);

    for (int i = 0; i < buttonCount; i++) {
        rect.setX((int)(this->x + space));
        rect.setY((int)yPosIter);
        rect.setHeight((int)length);
        rect.setWidth((int)length);
        button[i]->updateArea(rect);
        yPosIter += (length + space);
    }
}

//! Inits all needed variables and creates objects.
void QuickMenubar::init()
{
    this->lastPressedButton = 0;
    enablePoiEmit = false;
    //creates buttons. Measures y -position for them. ------ADJUST THIS LATER
    QRect rect;
    this->measureButtons();
    qreal space = length * 0.1;
    this->height = buttonCount * length;

    //calculate iters value 2 buttons upward to find starting y -position.
    int yPosIter = (int)(this->y + this->height / 2 - (1.5 * space) - (2 * length));

    //set filenames for the QString array.
    iconFileName[0] = new QString(zoomInButtonIcon);
    iconFileName[1] = new QString(zoomOutButtonIcon);
    iconFileName[2] = new QString(zoomPoiButtonIcon);
    iconFileName[3] = new QString(zoomTrackButtonIcon);

    for (int i = 0;i < buttonCount;i++) {
        rect.setX((int)space);
        rect.setY(yPosIter);
        rect.setHeight((int)length);
        rect.setWidth((int)length);
        button[i] = new QuickMenuButton(rect,
                                        *iconFileName[i], *backgroundColor);
        yPosIter += (int)(length + space);
    }
}

//! Sets background color for the quickmenu.
void QuickMenubar::setBackgroundColor(const QColor &color)
{
    delete backgroundColor;
    backgroundColor = new QColor(color);
}

//! Sets tracking button to "ON-mode".
void QuickMenubar::setTrackingSelected(bool mode)
{
    if (mode == true) {
        button[buttonTrack]->setMode(QIcon::Selected);
    }
    else if (mode == false) {
        button[buttonTrack]->setMode(QIcon::Normal);
    }
}
