/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef PREFERENCESDIALOG_H
#define PREFERENCESDIALOG_H

#include <QColor>
#include <QString>
#include <QVector>
#include "definitions.h"
#include "maprepository.h"
#include "ui_preferences.h"

class QSpinBox;
class QPushButton;
class QSettings;

/*!
 * Class PreferencesDialog displays a dialog where user can set up
 * various program parameters.
 */
class PreferencesDialog : public QDialog, Ui::PreferencesDialog
{
    Q_OBJECT

public:
    /*!
     * \brief Constructor.
     *
     * Constructs a PreferencesDialog object. The parameters are used to get
     * the current values of the application settings, which are placed
     * in the dialog's fields.
     *
     * \param parent The parent widget.
     * \param maxImageSpace The map image directory size limit.
     * \param imageRemoveAmount The number of images to remove when map image
     * directory size limit is reached.
     * \param aMapImageDirectory The directory to save the map images in.
     * \param repositories A reference to the list of map repositories.
     * \param activeRepository The currently used map repository.
     * \param routeLineColor The color of the line used to draw routes
     * and tracks.
     * \param routeLineThickness The thickness of the route/track line.
     * \param routeZoomLevel The default zoom level used with track drawing.
     * \param maxZoomLevel The maximum available zoom level.
     * \param advancedMode The advanced user mode setting.
     */
    PreferencesDialog(QWidget *parent,
                      qint64 maxImageSpace,
                      int imageRemoveAmount,
                      QString aMapImageDirectory,
                      QVector<MapRepository> &repositories,
                      int activeRepository,
                      QColor routeLineColor,
                      int routeLineThickness,
                      int routeZoomLevel,
                      int maxZoomLevel,
                      bool advancedMode);

    //! Destructor.
    ~PreferencesDialog();

    /*!
     * Get the contents of the maximum image space field of the dialog.
     *
     * \return The maximum image space in megabytes.
     */
    int getMaxImageSpace();

    /*!
     * Get the contents of the automatic image removal field of the dialog.
     *
     * \return The amount of images for automatic image removal.
     */
    int getImageRemoveAmount();

    /*!
     * Get the contents of the map image directory field of the dialog.
     *
     * \return The map image directory path.
     */
    QString getNewDirectory();

    /*!
     * Get the contents of the selected repository field of the dialog.
     *
     * \return The currently selected repository.
     */
    int getActiveRepository();

    /*!
     * Get the contents of the route line color field of the dialog.
     *
     * \return The route line color.
     */
    QColor getRouteLineColor();

    /*!
     * Get the contents of the route line thickness field of the dialog.
     *
     * \return The route line thickness.
     */
    int getRouteLineThickness();

    /*!
     * Get the contents of the route zoom level field of the dialog.
     *
     * Returns the route zoom level.
     */
    int getRouteZoomlevel() const { return defaultZoomLevelBox->value(); }

    /*!
     * Get the contents of the POI alarm distance field of the dialog.
     *
     * \return The distance to POI of POI alarm.
     */
    int getDistanceOfPoiAlarm();

    /*!
     * Get the contents of the number of nearest POIs field of the dialog.
     *
     * \return The count of POIs to be showed of POI alarm.
     */
    int getMaxCountOfPois();


    /*!
     * Get the contents of the advanced user mode field of the dialog.
     *
     * \return The advanced user mode setting.
     */
    bool getAdvancedMode() const;

    //! For loading settings value to POI alarm ui.
    QSettings *settings;

private:
    const QString spaceCharacter;
    const QString emptyString;
    const QString dbFileSuffix;
    const QString urlHttpPrefix;
    const QString white;
    const QString black;
    const QString red;
    const QString darkRed;
    const QString green;
    const QString darkGreen;
    const QString blue;
    const QString darkBlue;
    const QString cyan;
    const QString darkCyan;
    const QString magenta;
    const QString darkMagenta;
    const QString yellow;
    const QString darkYellow;

    //! reference to current map image directory
    QString mapImageDirectory;

    /*!
     * A reference to the list of map repositories,
     * which is a member of the QtMapper class.
     */
    QVector<MapRepository> &repositories;

    //! Add the list of map repositories into the repository combo box.
    void addRepositories();

    //! Load the default map repositories into the repository list.
    void resetRepositoriesToDefaults();

    /*!
     * Adds items to the line color combo box.
     *
     * \param selectedColor The currently used color.
     */
    void populateLineColorComboBox(QColor selectedColor);

    //! Loads values from settings to POI alarm Ui.
    void loadValuesToPoiAlarmUi();

private slots:
    //! Display a dialog for choosing map image directory
    void chooseImageDirectory();

    //! The reset repositories button was clicked.
    void defaultRepositoriesButtonClicked();

    //! The add repository button was clicked.
    void addRepositoryButtonClicked();

    //! The edit repository button was clicked.
    void editRepositoryButtonClicked();

    //! The remove repository button was clicked.
    void removeRepositoryButtonClicked();

    /*!
     * Enable the repository add/edit/remove buttons based on the combo
     * box contents.
     *
     * \param index The index of the currently selected item in the combo box.
     */
    void enableRepositoryManagementButtons(int index);

    /*!
     * Show or hide the advanced controls.
     *
     * \param mode The value of the advanced mode check box.
     */
    void showAdvancedModeControls(int mode);
};

#endif
