/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QFileDialog>
#include <QLineEdit>
#include <QMessageBox>
#include <QSettings>
#include "definitions.h"
#include "editrepositorydialog.h"
#include "preferencesdialog.h"

//! Constructor.
PreferencesDialog::PreferencesDialog(QWidget *parent,
                                     qint64 maxImageSpace,
                                     int imageRemoveAmount,
                                     QString aMapImageDirectory,
                                     QVector<MapRepository> &repositories,
                                     int activeRepository,
                                     QColor routeLineColor,
                                     int routeLineThickness,
                                     int routeZoomLevel,
                                     int maxZoomLevel,
                                     bool advancedMode) :
        QDialog(parent),
        spaceCharacter(" "),
        emptyString(""),
        dbFileSuffix(".db"),
        urlHttpPrefix("http://"),
        white(tr("White")),
        black(tr("Black")),
        red(tr("Red")),
        darkRed(tr("Dark red")),
        green(tr("Green")),
        darkGreen(tr("Dark green")),
        blue(tr("Blue")),
        darkBlue(tr("Dark blue")),
        cyan(tr("Cyan")),
        darkCyan(tr("Dark cyan")),
        magenta(tr("Magenta")),
        darkMagenta(tr("Dark magenta")),
        yellow(tr("Yellow")),
        darkYellow(tr("Dark yellow")),
        mapImageDirectory(aMapImageDirectory),
        repositories(repositories)
{
    setupUi(this);

    // maxImageSpace is in bytes, so it must be converted to Mbs.
    const int space = maxImageSpace / 1024 / 1024;

    addRepositories();
    comboRepositories->setCurrentIndex(activeRepository);

    populateLineColorComboBox(routeLineColor);
    lineThicknessBox->setValue(routeLineThickness);
    defaultZoomLevelBox->setValue(routeZoomLevel);
    defaultZoomLevelBox->setMaximum(maxZoomLevel);

    maxImagesSpinBox->setValue(space);
    removeImagesSpinBox->setValue(imageRemoveAmount);
    directoryLineEdit->setText(mapImageDirectory);

    connect(advancedModeBox, SIGNAL(stateChanged(int)),
            this, SLOT(showAdvancedModeControls(int)));
    connect(directoryButton, SIGNAL(clicked()),
            this, SLOT(chooseImageDirectory()));
    connect(defaultRepositoriesButton, SIGNAL(clicked()),
            this, SLOT(defaultRepositoriesButtonClicked()));
    connect(addRepositoryButton, SIGNAL(clicked()),
            this, SLOT(addRepositoryButtonClicked()));
    connect(editRepositoryButton, SIGNAL(clicked()),
            this, SLOT(editRepositoryButtonClicked()));
    connect(removeRepositoryButton, SIGNAL(clicked()),
            this, SLOT(removeRepositoryButtonClicked()));
    connect(comboRepositories, SIGNAL(currentIndexChanged(int)),
            this, SLOT(enableRepositoryManagementButtons(int)));

    advancedModeBox->setChecked(advancedMode);
    showAdvancedModeControls(advancedModeBox->checkState());

    loadValuesToPoiAlarmUi();
}

//! Destructor.
PreferencesDialog::~PreferencesDialog()
{
}

//! Get the contents of the maximum image space field of the dialog.
int PreferencesDialog::getMaxImageSpace()
{
    return maxImagesSpinBox->value();
}


//! Get the contents of the automatic image removal field of the dialog.
int PreferencesDialog::getImageRemoveAmount()
{
    return removeImagesSpinBox->value();
}

//! Get the contents of the POI alarm distance field of the dialog.
int PreferencesDialog::getDistanceOfPoiAlarm()
{
    return spinBoxDistance->value();
}

//! Get the contents of the number of nearest POIs field of the dialog.
int PreferencesDialog::getMaxCountOfPois()
{
    return spinBoxCountOfPois->value();
}

//! Loads values from settings to POI alarm Ui.
void PreferencesDialog::loadValuesToPoiAlarmUi()
{
    //Set settings values to spinBox
    settings = settings = new QSettings(settingOrganizationName,
                                        settingApplicationName);

    if (settings->contains(settingCountNearestPois)) {
        spinBoxCountOfPois->setValue(settings->value(settingCountNearestPois)
                                     .toInt());
    }
    else {
        spinBoxCountOfPois->setValue(defaultCountNearestPois);
    }

    if (settings->contains(settingDistanceToAlarm)) {
        spinBoxDistance->setValue(settings->value(settingDistanceToAlarm)
                                  .toInt());
    }
    else {
        spinBoxDistance->setValue(defaultDistanceToAlarm);
    }


    delete settings;
}

//! Display a dialog for choosing map image directory
void PreferencesDialog::chooseImageDirectory()
{
    QString dir = QFileDialog::getExistingDirectory(this,
            tr("Choose directory for map images"),
            mapImageDirectory,
            QFileDialog::ShowDirsOnly);

    // if user clicks ok, set the new directory
    if (!dir.isEmpty()) {
        mapImageDirectory = dir;
        directoryLineEdit->setText(dir);
    }
}

//! Get the contents of the map image directory field of the dialog.
QString PreferencesDialog::getNewDirectory()
{
    return mapImageDirectory;
}

//! Get the contents of the selected repository field of the dialog.
int PreferencesDialog::getActiveRepository()
{
    return comboRepositories->currentIndex();
}

//! Add a list of repositories into the repository combo box.
void PreferencesDialog::addRepositories()
{
    comboRepositories->clear();

    for (int i = 0; i < repositories.size(); i++) {
        comboRepositories->addItem(repositories.at(i).getName());
    }
}

//! The reset repositories button was clicked.
void PreferencesDialog::defaultRepositoriesButtonClicked()
{
    // Show a confirmation dialog.
    int result = QMessageBox::warning(this, tr("Reset Repositories"),
                                      tr("This will delete all saved map "
                                         "repositories and replace them with "
                                         "the default repositories.\n\n"
                                         "Do you want to continue?"),
                                      QMessageBox::Ok | QMessageBox::Cancel,
                                      QMessageBox::Cancel);

    if (result == QMessageBox::Ok) {
        resetRepositoriesToDefaults();

        // Refresh the combo box contents.
        addRepositories();
    }
}

//! The add repository button was clicked.
void PreferencesDialog::addRepositoryButtonClicked()
{
    EditRepositoryDialog *dialog = new EditRepositoryDialog(this);
    dialog->setWindowTitle(tr("Add Repository"));
    dialog->disableOkButton();

    if (dialog->exec() == QDialog::Accepted) {
        // Get the values from the dialog's text fields.
        QString name(dialog->nameText());
        QString url(dialog->urlText());
        QString imageNameFormat(dialog->imageNameFormatText());

        QString dbName(name.toLower());
        dbName.replace(spaceCharacter, emptyString);
        dbName.append(dbFileSuffix);

        // Add a new repository to the list.
        MapRepository newRepository(name, url, imageNameFormat, dbName,
                                    repositoryTypeXYZoom);
        repositories.append(newRepository);

        // Refresh the combo box contents.
        addRepositories();

        // Select the newly created repository in the combo box.
        comboRepositories->setCurrentIndex(comboRepositories->count() - 1);
    }

    delete dialog;
}

//! The edit repository button was clicked.
void PreferencesDialog::editRepositoryButtonClicked()
{
    EditRepositoryDialog *dialog = new EditRepositoryDialog(this);
    int selectedRepository = comboRepositories->currentIndex();

    // Put the repository info in the dialog's text fields.
    dialog->setNameText(repositories.at(selectedRepository).getName());
    dialog->setUrlText(repositories.at(selectedRepository).getUrl());
    dialog->setImageNameFormatText(repositories.at(selectedRepository)
                                   .getImageNameFormat());

    if (dialog->exec() == QDialog::Accepted) {
        // Update the repository info with the values from the dialog.
        repositories[selectedRepository].setName(dialog->nameText());
        repositories[selectedRepository].setUrl(dialog->urlText());
        repositories[selectedRepository]
                .setImageNameFormat(dialog->imageNameFormatText());

        QString dbName(dialog->nameText().toLower());
        dbName.replace(spaceCharacter, emptyString);
        dbName.append(dbFileSuffix);
        repositories[selectedRepository].setDatabase(dbName);

        // Refresh the combo box contents.
        addRepositories();

        // Select the newly created repository in the combo box.
        comboRepositories->setCurrentIndex(selectedRepository);
    }

    delete dialog;
}

//! The remove repository button was clicked.
void PreferencesDialog::removeRepositoryButtonClicked()
{
    int selectedRepository = comboRepositories->currentIndex();
    QString confirmationText(tr("Do you really want to "
                                "remove the map repository\n\"%1\"?")
                             .arg(repositories[selectedRepository].getName()));

    // Show a confirmation dialog.
    int result = QMessageBox::warning(this, tr("Remove Repository"),
                                      confirmationText,
                                      QMessageBox::Ok | QMessageBox::Cancel,
                                      QMessageBox::Cancel);

    if (result == QMessageBox::Ok) {
        repositories.remove(selectedRepository);

        // Refresh the combo box contents.
        addRepositories();
    }
}

//! Enable repository add/edit/remove buttons based on the combo box contents.
void PreferencesDialog::enableRepositoryManagementButtons(int index)
{
    const int comboBoxEmpty = -1;

    if (index == comboBoxEmpty) {
        removeRepositoryButton->setEnabled(false);
        editRepositoryButton->setEnabled(false);
    }
    else {
        removeRepositoryButton->setEnabled(true);
        editRepositoryButton->setEnabled(true);
    }
}

//! Load the default map repositories into the repository list.
void PreferencesDialog::resetRepositoriesToDefaults()
{
    repositories.clear();
    repositories.append(openStreetMap);
}

//! Adds items to the line color combo box.
void PreferencesDialog::populateLineColorComboBox(QColor selectedColor)
{
    lineColorBox->addItem(QIcon(colorWhiteResource), white);
    lineColorBox->addItem(QIcon(colorBlackResource), black);
    lineColorBox->addItem(QIcon(colorRedResource), red);
    lineColorBox->addItem(QIcon(colorDarkRedResource), darkRed);
    lineColorBox->addItem(QIcon(colorGreenResource), green);
    lineColorBox->addItem(QIcon(colorDarkGreenResource), darkGreen);
    lineColorBox->addItem(QIcon(colorBlueResource), blue);
    lineColorBox->addItem(QIcon(colorDarkBlueResource), darkBlue);
    lineColorBox->addItem(QIcon(colorCyanResource), cyan);
    lineColorBox->addItem(QIcon(colorDarkCyanResource), darkCyan);
    lineColorBox->addItem(QIcon(colorMagentaResource), magenta);
    lineColorBox->addItem(QIcon(colorDarkMagentaResource), darkMagenta);
    lineColorBox->addItem(QIcon(colorYellowResource), yellow);
    lineColorBox->addItem(QIcon(colorDarkYellowResource), darkYellow);

    if (selectedColor == Qt::white) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(white));
    }
    else if (selectedColor == Qt::black) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(black));
    }
    else if (selectedColor == Qt::red) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(red));
    }
    else if (selectedColor == Qt::darkRed) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(darkRed));
    }
    else if (selectedColor == Qt::green) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(green));
    }
    else if (selectedColor == Qt::darkGreen) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(darkGreen));
    }
    else if (selectedColor == Qt::blue) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(blue));
    }
    else if (selectedColor == Qt::darkBlue) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(darkBlue));
    }
    else if (selectedColor == Qt::cyan) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(cyan));
    }
    else if (selectedColor == Qt::darkCyan) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(darkCyan));
    }
    else if (selectedColor == Qt::magenta) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(magenta));
    }
    else if (selectedColor == Qt::darkMagenta) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(darkMagenta));
    }
    else if (selectedColor == Qt::yellow) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(yellow));
    }
    else if (selectedColor == Qt::darkYellow) {
        lineColorBox->setCurrentIndex(lineColorBox->findText(darkYellow));
    }
}

//! Get the contents of the route line color field of the dialog.
QColor PreferencesDialog::getRouteLineColor()
{
    QColor color;
    QString selectedItemText = lineColorBox
                               ->itemText(lineColorBox->currentIndex());

    if (selectedItemText == white) {
        color = Qt::white;
    }
    else if (selectedItemText == black) {
        color = Qt::black;
    }
    else if (selectedItemText == red) {
        color = Qt::red;
    }
    else if (selectedItemText == darkRed) {
        color = Qt::darkRed;
    }
    else if (selectedItemText == green) {
        color = Qt::green;
    }
    else if (selectedItemText == darkGreen) {
        color = Qt::darkGreen;
    }
    else if (selectedItemText == blue) {
        color = Qt::blue;
    }
    else if (selectedItemText == darkBlue) {
        color = Qt::darkBlue;
    }
    else if (selectedItemText == cyan) {
        color = Qt::cyan;
    }
    else if (selectedItemText == darkCyan) {
        color = Qt::darkCyan;
    }
    else if (selectedItemText == magenta) {
        color = Qt::magenta;
    }
    else if (selectedItemText == darkMagenta) {
        color = Qt::darkMagenta;
    }
    else if (selectedItemText == yellow) {
        color = Qt::yellow;
    }
    else if (selectedItemText == darkYellow) {
        color = Qt::darkYellow;
    }

    return color;
}

//! Get the contents of the route line thickness field of the dialog.
int PreferencesDialog::getRouteLineThickness()
{
    return lineThicknessBox->value();
}

//! Get the contents of the advanced user mode field of the dialog.
bool PreferencesDialog::getAdvancedMode() const
{
    return advancedModeBox->isChecked();
}

//! Show or hide the advanced controls.
void PreferencesDialog::showAdvancedModeControls(int mode)
{
    const int checked = 2;
    const QString mapRepositoriesTabLabel("Map Repositories");
    const int mapRepositoriesTabIndex = 1;

    if (mode == checked) {
        tabWidget->insertTab(mapRepositoriesTabIndex,
                             mapRepositoriesTab,
                             mapRepositoriesTabLabel);

        removeImagesLabel->show();
        removeImagesSpinBox->show();
    }
    else {
        tabWidget->removeTab(mapRepositoriesTabIndex);

        removeImagesLabel->hide();
        removeImagesSpinBox->hide();
    }
}
