/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __POILISTDIALOG_H__
#define __POILISTDIALOG_H__

#include "ui_poilist.h"
#include "QtMapper.h"
#include <QList>
#include <QVector>

class Poi;
class PoiCategory;
class PoiDatabase;
class QStandardItemModel;

//! A dialog to display a list of saved points of interest.
class PoiListDialog : public QDialog, public Ui::PoiListDialog
{
    Q_OBJECT

public:
    /*!
     * Constructor.
     *
     * \param radio The selected radio button in the dialog.
     * \param mainWindow A reference to the application main window.
     * \param latitude Current latitude.
     * \param longitude Current longitude.
     * \param limit How many POIs to get from the database.
     * \param categoryId Selected POI category.
     * \param parent The parent widget of the dialog.
     */
    PoiListDialog(qint32 radio,
                  QtMapper &mainWindow,
                  qreal latitude,
                  qreal longitude,
                  qint32 limit,
                  qint32 categoryId,
                  bool validGpsDataAvailable,
                  QWidget *parent = 0);

    /*!
     * Constructor.
     *
     * \param radio The selected radio button in the dialog.
     * \param mainWindow A reference to the application main window.
     * \param latitude Current latitude.
     * \param longitude Current longitude.
     * \param parent The parent widget of the dialog.
     */
    PoiListDialog(qint32 radio,
                  QtMapper &mainWindow,
                  qreal latitude,
                  qreal longitude,
                  bool validGpsDataAvailable,
                  QWidget *parent = 0);

    /*!
     * Constructor.
     *
     * \param radio The selected radio button in the dialog.
     * \param keyWord The POI search keyword.
     * \param mainWindow A reference to the application main window.
     * \param latitude Current latitude.
     * \param longitude Current longitude.
     * \param parent The parent widget of the dialog.
     */
    PoiListDialog(qint32 radio,
                  QString keyWord,
                  QtMapper &mainWindow,
                  qreal latitude,
                  qreal longitude,
                  bool validGpsDataAvailable,
                  QWidget *parent = 0);

    //! Destructor.
    ~PoiListDialog();

private:
    //! Whether valid gps location data is available
    bool validGpsDataAvailable;

    //! Earth radius in kilometers
    const int earthRadiusInKilometres;

    //! width of 1. column of view
    const int column1Width;

    //! width of 2. column of view
    const int column2Width;

    //! width of 3. column of view
    const int column3Width;

    //! width of 4. column of view
    const int column4Width;

    //! width of 5. column of view
    const int column5Width;

    /*!
     * Pointer to caller dialog. Used when "Goto" is
     * pressed to close this and caller Dialog.
     */
    QWidget *parent;

    //! Parameter from Browse Pois Dialog
    qint32 selectedOption;

    //! POI search keyword.
    QString keyWord;

    //! Reference to main window
    QtMapper & mainWindow;

    //! Container for POI category data
    QVector<PoiCategory> categories;

    //! Container for POI data
    QVector<Poi> pois;

   	//! Model of POI data
   	QStandardItemModel * model;  
   	
   	//! Pointer to POI database
   	PoiDatabase *poiDb; 	
   	
   	//! Current latitude
    qreal latitude;

    //! Current longitude
    qreal longitude;

    //! Limit of POIs to be loaded
    qint32 limit;

    //! Category ID of POIs to be showed in POI List Dialog
    qint32 categoryId;

    /*!
     * Calculates distance from current location to POI's location.
     *
     * \param startLatitude Latitude at the first location
     * \param startLongitude Longitude at the first location.
     * \param endLatitude Latitude at the second location,
     * \param endLongitude Longitude at the second location.
     */
    qreal distanceBetweenCoordinates(qreal startLatitude,
                                     qreal startLongitude,
                                     qreal endLatitude,
                                     qreal endLongitude);

    /*!
     * Calculates bearing from current location to POI's location.
     *
     * \param startLatitude Latitude at the first location
     * \param startLongitude Longitude at the first location.
     * \param endLatitude Latitude at the second location,
     * \param endLongitude Longitude at the second location.
     */
    qreal bearing(qreal startLatitude,
                  qreal startLongitude,
                  qreal endLatitude,
                  qreal endLongitude);

    //! Connect signals to slots.
    void connectSignals();

private slots:
    /*!
     * Shows dialog Delete POI and handles updating of database
     * when POI data is deleted.
     */
    void showDeletePoiDialog();

    /*!
     * Shows dialog Add POI and handles updating of database when
     * POI data is added.
     */
    void showAddPoiDialog();

    /*!
     * Shows dialog Edit POI and handles updating of database when
     * POI data is changed.
     */
    void showEditPoiDialog();

    /*!
     * Sets the data of POIs to view.
     */
    void setData();

    /*!
     * Sets the View's center point to selected POI's location.
     */
    void goTo();

    //! Export selected POIs.
    void exportPois();

    /*!
     * Create a GPX file with POI information.
     *
     * \return The file contents.
     */
    QString createGpxFile(QList<Poi> &poisToExport);
};

#endif
