/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QAbstractItemView>
#include <QDebug>
#include <QDomDocument>
#include <QDoubleValidator>
#include <QFileDialog>
#include <QMessageBox>
#include <QStandardItem>
#include <QStandardItemModel>
#include <QValidator>
#include <math.h>
#include "definitions.h"
#include "addpoidialog.h"
#include "editpoidialog.h"
#include "poidatabase.h"
#include "poilistdialog.h"
#include "poi.h"
#include "poicategory.h"

//! Constructor.
PoiListDialog::PoiListDialog(qint32 selectedOption,
                             QtMapper &mainWindow,
                             qreal latitude,
                             qreal longitude,
                             qint32 limit,
                             qint32 categoryId,
                             bool validGpsDataAvailable,
                             QWidget *parent) :
        QDialog(parent),
        earthRadiusInKilometres(6371),
        column1Width(80),
        column2Width(150),
        column3Width(120),
        column4Width(100),
        column5Width(160),
        selectedOption(selectedOption),
        mainWindow(mainWindow),
        latitude(latitude),
        longitude(longitude),
        limit(limit),
        categoryId(categoryId)
{
    this->parent = parent;
    setupUi(this);
    connectSignals();
    poiDb = mainWindow.getPoiDb();
    
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }

    this->validGpsDataAvailable = validGpsDataAvailable;

    setData();
}

//! Constructor.
PoiListDialog::PoiListDialog(qint32 selectedOption,
                             QtMapper &mainWindow,
                             qreal latitude,
                             qreal longitude,
                             bool validGpsDataAvailable,
                             QWidget *parent) :
        QDialog(parent),
        earthRadiusInKilometres(6371),
        column1Width(80),
        column2Width(150),
        column3Width(120),
        column4Width(100),
        column5Width(150),
        selectedOption(selectedOption),
        mainWindow(mainWindow),
        latitude(latitude),
        longitude(longitude)
{
    this->parent = parent;
    setupUi(this);
    connectSignals();
    poiDb = mainWindow.getPoiDb();
    
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }

    this->validGpsDataAvailable = validGpsDataAvailable;

    setData();
}

//! Constructor.
PoiListDialog::PoiListDialog(qint32 selectedOption,
                             QString keyWord,
                             QtMapper &mainWindow,
                             qreal latitude,
                             qreal longitude,
                             bool validGpsDataAvailable,
                             QWidget *parent) :
    QDialog(parent),
    earthRadiusInKilometres(6371),
    column1Width(80),
    column2Width(150),
    column3Width(120),
    column4Width(100),
    column5Width(150),
    selectedOption(selectedOption),
    keyWord(keyWord),
    mainWindow(mainWindow),
    latitude(latitude),
    longitude(longitude)
{
    this->parent = parent;
    setupUi(this);
    connectSignals();
    poiDb = mainWindow.getPoiDb();
    
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }

    this->validGpsDataAvailable = validGpsDataAvailable;

    setData();
}

//! Connect signals to slots.
void PoiListDialog::connectSignals()
{
    connect(pushButtonEdit, SIGNAL(clicked()),
            this, SLOT(showEditPoiDialog()));
    connect(pushButtonAdd, SIGNAL(clicked()),
            this, SLOT(showAddPoiDialog()));
    connect(pushButtonDelete, SIGNAL(clicked()),
            this, SLOT(showDeletePoiDialog()));
    connect(pushButtonClose, SIGNAL(clicked()),
            this, SLOT(accept()));
    connect(pushButtonGoTo, SIGNAL(clicked()),
            this, SLOT(goTo()));
    connect(pushButtonExport, SIGNAL(clicked()),
            this, SLOT(exportPois()));
}

//! Sets the data of POIs to view.
void PoiListDialog::setData()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }   
    
    if (selectedOption == 0) {
        poiDb->getPois(pois, latitude, longitude, limit, categoryId);
    }
    else if (selectedOption == 1) {
        poiDb->getPoisByKeyWord(pois, keyWord);
    }
    else {
        poiDb->getAllPois(pois);
    }

    model = new QStandardItemModel(this);

    QStandardItem *item0 = new QStandardItem(QString(tr("Select")));
    QStandardItem *item1 = new QStandardItem(QString(tr("Category")));
    QStandardItem *item2 = new QStandardItem(QString (tr("Distance")));
    QStandardItem *item3 = new QStandardItem(QString (tr("Bearing")));
    QStandardItem *item4 = new QStandardItem(QString (tr("Label")));

    model->setHorizontalHeaderItem(0, item0);
    model->setHorizontalHeaderItem(2, item1);
    model->setHorizontalHeaderItem(3, item2);
    model->setHorizontalHeaderItem(4, item3);
    model->setHorizontalHeaderItem(1, item4);

    for (qint32 i = 0; i < pois.count(); i++) {

        QStandardItem *item0 = new QStandardItem(QString(""));
        item0->setCheckState(Qt::Unchecked);
        item0->setCheckable(true);

        QStandardItem *item1 = new QStandardItem(QString(pois.at(i)
                                                        .getCategory()
                                                        .getLabel()));

        QStandardItem *item2;
        QStandardItem *item3;

        if (validGpsDataAvailable) {
            item2 = new QStandardItem(QString("%1")
                    .arg(distanceBetweenCoordinates(latitude, longitude,
                                                   pois.at(i).getLatitude(),
                                                   pois.at(i).getLongitude())));

            item3 = new QStandardItem(QString("%1")
                .arg(bearing(latitude, longitude, pois.at(i).getLatitude(),
                             pois.at(i).getLongitude())));

        }
        else {
            item2 = new QStandardItem(QString(" - "));
            item3 = new QStandardItem(QString(" - "));
        }

        QStandardItem *item4 = new QStandardItem(QString(pois.at(i)
                .getLabel()));

        item1->setEditable(false);
        item2->setEditable(false);
        item3->setEditable(false);
        item4->setEditable(false);

        model->setItem(i, 0, item0);
        model->setItem(i, 2, item1);
        model->setItem(i, 3, item2);
        model->setItem(i, 4, item3);
        model->setItem(i, 1, item4);
    }

    tableView->setModel(model);
    tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
    tableView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    tableView->setColumnWidth(0, column1Width);
    tableView->setColumnWidth(2, column2Width);
    tableView->setColumnWidth(3, column3Width);
    tableView->setColumnWidth(4, column4Width);
    tableView->setColumnWidth(1, column5Width);

}

/*!
* Shows dialog Edit POI and handles updating of database when POI data
* is changed.
*/
void PoiListDialog::showEditPoiDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    categories.clear();
    poiDb->getPoiCategories(categories);

    EditPoiDialog ep(mainWindow, this);
    
    qint32 i = tableView->currentIndex().row();

    if (i < 0 || i >= pois.count()) {
            i = 0;
            return;
    }

    QDoubleValidator latitudeValidator(-90.000000, 90.000000, 6, this);
    QDoubleValidator longitudeValidator(-180.000000, 180.000000, 6, this);

    ep.lineEditLabel->setText(pois.at(i).getLabel());
    ep.textEditDescription->setText(pois.at(i).getDescription());
    ep.lineEditLatitude->setText(QString("%1").arg(pois.at(i).getLatitude()));
    ep.lineEditLatitude->setValidator(&latitudeValidator);
    ep.lineEditLongitude->setText(QString("%1").arg(pois.at(i).getLongitude()));
    ep.lineEditLongitude->setValidator(&longitudeValidator);

    for (qint32 j = 0; j < categories.count(); j++) {
        ep.comboBoxCategory->addItem(categories.at(j).getLabel());
    }


    ep.comboBoxCategory->setCurrentIndex(ep.comboBoxCategory
            ->findText(pois.at(i).getCategory().getLabel()));

    if (ep.exec() == QDialog::Accepted) {
        pois[i].setLabel(ep.lineEditLabel->text());
        pois[i].setDescription(ep.textEditDescription->toPlainText());
        pois[i].setLatitude(ep.lineEditLatitude->text().toDouble());
        pois[i].setLongitude(ep.lineEditLongitude->text().toDouble());
        pois[i].setCategory(categories.at(ep.comboBoxCategory->currentIndex()));

        poiDb->updatePoi(pois[i]);
        pois.clear();
        setData();
        tableView->selectRow(i);
    }
}

//! Shows dialog Add POI and handles updating of database when POI data is added.
void PoiListDialog::showAddPoiDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    categories.clear();
    poiDb->getPoiCategories(categories);

    AddPoiDialog ap(mainWindow, this);

    for (qint32 i = 0; i < categories.count(); i++) {
        ap.comboBoxCategory->addItem(categories.at(i).getLabel());
    }

    if (ap.exec() == QDialog::Accepted) {
        ap.addPoiToDatabase();
        pois.clear();
        setData();
    }
}

/*!
 * Shows dialog Delete POI and handles updating of database when POI data
 * is deleted.
 */
void PoiListDialog::showDeletePoiDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    QMessageBox mb(this);

    if (pois.count()==0) {
        return;
    }

    mb.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
    mb.setWindowTitle(tr("Confirm Deletion"));
    QString text(tr("Remove "));
    qint32 countDeletedPois = 0;

    for (qint32 i = 0; i < pois.count(); i++) {
        if (model->item(i)->checkState() == Qt::Checked) {
            countDeletedPois++;
        }
    }

    text.append(QString("%1 ").arg(countDeletedPois));
    text.append(tr("POIs?\n"));
    mb.setText(text);

    if (countDeletedPois == 0) {
        return;
    }

    if (mb.exec() == QMessageBox::Ok) {
        for (qint32 i = 0; i < pois.count(); i++) {
            if (model->item(i)->checkState() == Qt::Checked) {
                poiDb->removePoi(pois.at(i).getPoiId());
            }
        }

        model->clear();
        pois.clear();
        setData();
    }
}

//! Calculates distance from current location to POI's location.
qreal PoiListDialog::distanceBetweenCoordinates(qreal startLatitude,
        qreal startLongitude,
        qreal endLatitude,
        qreal endLongitude)
{
    /* convert degrees to radians */
    startLatitude = startLatitude * pi / 180;
    startLongitude = startLongitude * pi / 180;
    endLatitude = endLatitude * pi / 180;
    endLongitude = endLongitude * pi / 180;

    qreal latitudeDifference = endLatitude - startLatitude;
    qreal longitudeDifference = endLongitude - startLongitude;

    qreal temp = pow(sin(latitudeDifference / 2), 2) +
                 cos(startLatitude) * cos(endLatitude) *
                 pow(sin(longitudeDifference / 2), 2);

    qreal distance = earthRadiusInKilometres * 2 * atan2(sqrt(temp),
                     sqrt(1 - temp));

    return distance;
}

//! Calculates bearing from current location to POI's location.
qreal PoiListDialog::bearing(qreal startLatitude,
                             qreal startLongitude,
                             qreal endLatitude,
                             qreal endLongitude)
{
    /* convert degrees to radians */
    startLatitude = startLatitude * pi / 180;
    startLongitude = startLongitude * pi / 180;
    endLatitude = endLatitude * pi / 180;
    endLongitude = endLongitude * pi / 180;

    qreal longitudeDifference = endLongitude - startLongitude;

    /* calculate the bearing */
    qreal y = sin(longitudeDifference) * cos(endLatitude);
    qreal x = cos(startLatitude) * sin(endLatitude) - sin(startLatitude) *
              cos(endLatitude) * cos(longitudeDifference);
    qreal bearing = atan2(y, x);

    /* convert bearing to degrees */
    bearing = bearing * 180 / pi;

    return bearing;
}

//! Sets the View's center point to selected POI's location.
void PoiListDialog::goTo()
{
    if (pois.count() == 0) {
        return;
    }

    qint32 i = tableView->currentIndex().row();

    if (i < 0 || i > pois.count()) {
        i = 0;
        return;
    }
    
    mainWindow.goToSelectedPoi(pois.at(i));
    this->parent->close();
    this->close();
}

//! Export selected POIs.
void PoiListDialog::exportPois()
{
    const QString fileExt(".gpx");
    QList<Poi> poisToExport;

    for (int i = 0; i < model->rowCount(); i++) {
        QStandardItem *item = model->item(i);

        if (item->isCheckable() && item->checkState() == Qt::Checked) {
            poisToExport.append(pois.at(i));
        }
    }

    if (poisToExport.isEmpty()) {
        return;
    }

    QString filename = QFileDialog::getSaveFileName(this, tr("Export POI"),
                       QDir::currentPath(),
                       tr("GPX files(*.gpx)"));
    if (!filename.endsWith(fileExt)) {
        filename.append(fileExt);
    }

    QFile file(filename);

    if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QString msg = QString(tr("Unable to save file:\n\n"))
                      .append(file.errorString());
        QMessageBox::critical(this, tr("Error"), msg, QMessageBox::Ok);
    }
    else {
        QString gpxData = createGpxFile(poisToExport);
        QTextStream out(&file);
        out << gpxData;
        file.close();
    }
}

//! Create a GPX file with POI information.
QString PoiListDialog::createGpxFile(QList<Poi> &poisToExport)
{
    const QString gpxElement("gpx");
    const QString versionAttr("version");
    const QString versionValue("1.1");
    const QString creatorAttr("creator");
    const QString creatorValue("QtMapper");
    const QString xmlnsAttr("xmlns");
    const QString xmlnsValue("http://www.topografix.com/GPX/1/1");
    const QString wptElement("wpt");
    const QString latAttr("lat");
    const QString lonAttr("lon");
    const QString nameElement("name");
    const QString descElement("desc");
    const int indentSize = 4;

    QDomDocument doc;

    QDomElement gpx = doc.createElement(gpxElement);
    gpx.setAttribute(versionAttr, versionValue);
    gpx.setAttribute(creatorAttr, creatorValue);
    gpx.setAttribute(xmlnsAttr, xmlnsValue);
    doc.appendChild(gpx);

    for (int i = 0; i < poisToExport.size(); i++) {
        Poi poi = poisToExport.at(i);
        QDomElement wpt = doc.createElement(wptElement);
        wpt.setAttribute(latAttr, QString("%1").arg(poi.getLatitude()));
        wpt.setAttribute(lonAttr, QString("%1").arg(poi.getLongitude()));
        gpx.appendChild(wpt);

        QDomElement name = doc.createElement(nameElement);
        QDomText nameText = doc.createTextNode(poi.getLabel());
        name.appendChild(nameText);
        wpt.appendChild(name);

        QDomElement desc = doc.createElement(descElement);
        QDomText descText = doc.createTextNode(poi.getDescription());
        desc.appendChild(descText);
        wpt.appendChild(desc);
    }

    return doc.toString(indentSize);
}

//! Destructor.
PoiListDialog::~PoiListDialog()
{
}
