/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __POIDATABASE_H__
#define __POIDATABASE_H__

#include <QVector>
#include <QSqlDatabase>

class Poi;
class PoiCategory;

/*!
 * Class for handling a database of POI data.
 */
class PoiDatabase
{
public:
    /*!
     * Constructor.
     */
    PoiDatabase();

    /*!
     * Destructor.
     */
    ~PoiDatabase();

    /*!
     * Opens a database connection.
     */
    bool createConnection();

    /*!
     * Closes a database connection.
     */
    void closeConnection() { this->db.close(); }

    /*!
     * Get database connection state.
     *
     * \return True if connection open, false otherwise
     */
    bool isConnected() const { return this->db.isOpen(); }

    /*!
     * Inserts a POI.
     * \param poi POI (Point of Interest).
     */
    bool insertPoi(Poi poi);

    /*!
     * Updates a POI to database with new POI data given in parameter.
     * \param poi Updated POI.
     */
    bool updatePoi(Poi poi);

    /*!
     * Removes POI from database by ID given in parameter.
     * \param poiId ID of POI.
     */
    bool removePoi(quint32 poiId);
    /*!
     * Removes POIs from database by category ID given in parameter.
     * \param categoryId ID of POI category.
     */
    bool removePoiByCategoryId(quint32 categoryId);

    /*!
     * Inserts some default POI categories to database.
     */
    void insertDefaultCategories();

    /*!
     * Inserts POI category to database.
     * \param category Poi category.
     */
    bool insertPoiCategory(PoiCategory category);

    /*!
     * Updates POI category to database with new data given in parameter.
     * \param category Poi category.
     */
    bool updatePoiCategory(PoiCategory category);

    /*!
     * Removes POI category from database by ID given in parameter.
     * \param categoryId ID of POI category.
     */

    bool removePoiCategory(quint32 categoryId);
    /*!
     * Sets all POI categories to the given vector.
     * \param categories Vector of POI categories.
     */
    bool getPoiCategories(QVector<PoiCategory> & categories);

    /*!
     * Returns POI category id matching label or if not found returns -1.
     * \param label Label of POI category.
     */
    qint32 getCategoryId(QString label);

    /*!
     * Gets POIs by latitude and longitude to vector from database.
     * \param pois Vector of POIs.
     * \param latitude Latitude of the current Gps position.
     * \param longitude Longitude of the current Gps position.
     * \param limit Maximum limit of the result of database query.
     * \param categoryId Category id of the POIs to be searched.
     */
    bool getPois(QVector<Poi> & pois, double latitude,
                 double longitude, quint32 limit, qint32 categoryId);

    /*!
     * Sets all POIs from database to vector.
     * \param pois Vector of POIs.
     */
    bool getAllPois(QVector<Poi> & pois);

    /*!
     * Gets POIs by key word to vector from database.
     * \param pois Vector of POIs.
     * \param keyWord Key word for select query for searching POIs.
     */
    bool getPoisByKeyWord(QVector<Poi> & pois, QString keyWord);

private:
    //! Database object.
    QSqlDatabase db;

    /*!
     * Creates POI table to database.
     */
    bool createPoiTable();

    /*!
     * Creates POI categories table to database.
     */
    bool createPoiCategoryTable();

    //! Database type
    const QString databaseType;

    //! Connection name
    const QString connectionName;
};

#endif
