/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QSqlQuery>
#include <QSqlRecord>
#include <QDebug>
#include <QSqlError>
#include <QString>
#include <QVariant>
#include "poidatabase.h"
#include "poi.h"
#include "poicategory.h"
#include <stdlib.h>
#include <QDir>

//! Constructor
PoiDatabase::PoiDatabase():
    databaseType("QSQLITE"),
    connectionName("poiDb")
{
    const QString databasePath(QString("%1/.qtmapper/poidatabase/").arg(getenv ("HOME")));
    const QString databaseName(QString("%1%2").arg(databasePath).arg("PoiDatabase.db"));

    QDir *dir1 = new QDir(databasePath);

    if (!dir1->exists()) {
        QDir *dir2 = new QDir();
        dir2->mkpath(databasePath);
        delete dir2;
    }

    delete dir1;

    db = QSqlDatabase::addDatabase(databaseType, connectionName);
    db.setDatabaseName(databaseName);

    if (db.open()) {
        createPoiTable();
        createPoiCategoryTable();
        db.close();
    }
}

//! Opens a database connection.
bool PoiDatabase::createConnection()
{
    if (db.open()) {
        return true;
    }

    return false;
}

//! Creates POI table to database.
bool PoiDatabase::createPoiTable()
{
    QSqlQuery query(db);

    if (query.exec("create table poi (poiId integer PRIMARY KEY, "
                   "latitude real, longitude real, label text, "
                   "description text, categoryId integer)")) {
        return true;
    }

    return false;
}

//! Creates POI categories table to database.
bool PoiDatabase::createPoiCategoryTable()
{
    QSqlQuery query(db);

    if (query.exec("create table poicategory (categoryId integer "
                   "PRIMARY KEY, label text, description text, "
                   "enabled integer)")) {
        return true;
    }

    return false;
}

//! Inserts a POI.
bool PoiDatabase::insertPoi(Poi poi)
{
    QSqlQuery query(db);

    query.prepare("INSERT INTO poi (latitude, longitude, label, "
                  "description, categoryId) VALUES (?, ?, ?, ?, ?)");
    query.bindValue(0, poi.getLatitude());
    query.bindValue(1, poi.getLongitude());
    query.bindValue(2, poi.getLabel());
    query.bindValue(3, poi.getDescription());
    query.bindValue(4, poi.getCategory().getCategoryId());

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Updates a POI to database with new POI data given in parameter.
bool PoiDatabase::updatePoi(Poi poi)
{
    QSqlQuery query(db);

    query.prepare("update poi set latitude = ?, longitude = ?, label = ?, "
                  "description = ?, categoryId = ? where poiId = ?");
    query.bindValue(0, poi.getLatitude());
    query.bindValue(1, poi.getLongitude());
    query.bindValue(2, poi.getLabel());
    query.bindValue(3, poi.getDescription());
    query.bindValue(4, poi.getCategory().getCategoryId());
    query.bindValue(5, poi.getPoiId());

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Removes POI from database by ID given in parameter.
bool PoiDatabase::removePoi(quint32 poiId)
{
    QSqlQuery query(db);
    query.prepare("DELETE FROM poi WHERE poiId=?");
    query.bindValue(0, poiId);

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Removes POIs from database by category ID given in parameter.
bool PoiDatabase::removePoiByCategoryId(quint32 categoryId)
{
    QSqlQuery query(db);
    query.prepare("delete from poi where categoryId = ?");
    query.bindValue(0, categoryId);

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Inserts POI category to database.
bool PoiDatabase::insertPoiCategory(PoiCategory category)
{
    QSqlQuery query(db);

    query.prepare("INSERT INTO poicategory (label, description, enabled)"
                  "VALUES (?, ?, ?)");
    query.bindValue(0, category.getLabel());
    query.bindValue(1, category.getDescription());
    query.bindValue(2, category.getEnabled());

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Sets all POI categories to the given vector.
bool PoiDatabase::getPoiCategories(QVector<PoiCategory> & categories)
{
    QSqlQuery query(db);
    query.prepare("select * from poicategory");

    if (query.exec()) {
        while (query.next()) {
            qint32 categoryId = query.value(0).toInt();
            QString label = query.value(1).toString();
            QString description = query.value(2).toString();
            qint32 enabled = query.value(3).toInt();

            PoiCategory category;
            category.setCategoryId(categoryId);
            category.setLabel(label);
            category.setDescription(description);
            category.setEnabled(enabled);
            categories.append(category);
        }

        return true;
    }

    return false;
}

//! Updates POI category to database with new data given in parameter.
bool PoiDatabase::updatePoiCategory(PoiCategory category)
{
    QSqlQuery query(db);
    query.prepare("update poicategory set label = ?, description = ?,"
                  " enabled = ? where categoryId = ?");
    query.bindValue(0, category.getLabel());
    query.bindValue(1, category.getDescription());
    query.bindValue(2, category.getEnabled());
    query.bindValue(3, category.getCategoryId());

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Returns POI category id matching label or if not found returns -1.
qint32 PoiDatabase::getCategoryId(QString label)
{
    QSqlQuery query(db);
 
    query.prepare("select categoryId from poicategory"
                  " where label = ? ");
                
    query.bindValue(0, QString("%1").arg(label));
   
    if (query.exec()) {
        while (query.next()) {
           qint32 categoryId = query.value(0).toInt();
           return categoryId;
        }
    }

    return -1;
}

//! Removes POI category from database by ID given in parameter.
bool PoiDatabase::removePoiCategory(quint32 categoryId)
{
    QSqlQuery query(db);
    query.prepare("delete from poicategory where categoryId = ?");
    query.bindValue(0, categoryId);

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Gets POIs by latitude and longitude to vector from database.
bool PoiDatabase::getPois(QVector<Poi> & pois, double latitude,
                          double longitude, quint32 limit, qint32 categoryId)
{
    QSqlQuery query(db);
  
    if (categoryId > 0) {
        query.prepare("select poi.poiId, poi.latitude, poi.longitude, "
                      "poi.label, poi.description, poi.categoryId, "
                      "poicategory.label, poicategory.description, "
                      "poicategory.enabled from poi inner join poicategory "
                      "on poi.categoryId = poicategory.categoryId "
                      "where poi.categoryId = ? order by ((? - poi.latitude) "
                      "* (? - poi.latitude) + (? - poi.longitude) * "
                      "(? - poi.longitude)) limit ?");
        query.bindValue(0, QString("%1").arg(categoryId));
        query.bindValue(1, latitude);
        query.bindValue(2, latitude);
        query.bindValue(3, longitude);
        query.bindValue(4, longitude);
        query.bindValue(5, limit);
        
    }
    else {
        query.prepare("select poi.poiId, poi.latitude, poi.longitude, "
                      "poi.label, poi.description, poi.categoryId, "
                      "poicategory.label, poicategory.description, "
                      "poicategory.enabled from poi inner join poicategory "
                      "on poi.categoryId = poicategory.categoryId "
                      "order by ((? - poi.latitude) * (? - poi.latitude) "
                      "+ (? - poi.longitude) * (? - poi.longitude)) limit ?");
        query.bindValue(0, latitude);
        query.bindValue(1, latitude);
        query.bindValue(2, longitude);
        query.bindValue(3, longitude);
        query.bindValue(4, limit);
    }

    if (query.exec()) {
        while (query.next()) {
            qint32 poiId = query.value(0).toInt();
            double latitude = query.value(1).toDouble();
            double longitude = query.value(2).toDouble();
            QString poiLabel = query.value(3).toString();
            QString poiDescription = query.value(4).toString();
            qint32 categoryId = query.value(5).toInt();
            QString categoryLabel = query.value(6).toString();
            QString categoryDescription = query.value(7).toString();
            qint32 enabled = query.value(8).toInt();

            PoiCategory category;
            category.setCategoryId(categoryId);
            category.setLabel(categoryLabel);
            category.setDescription(categoryDescription);
            category.setEnabled(enabled);

            Poi poi;
            poi.setPoiId(poiId);
            poi.setLatitude(latitude);
            poi.setLongitude(longitude);
            poi.setLabel(poiLabel);
            poi.setDescription(poiDescription);
            poi.setCategory(category);

            pois.append(poi);
          
        }

        return true;
    }

    return false;
}

//! Sets all POIs from database to vector.
bool PoiDatabase::getAllPois(QVector<Poi> & pois)
{
    QSqlQuery query(db);
    query.prepare("select poi.poiId, poi.latitude, poi.longitude,"
                  " poi.label, poi.description, poi.categoryId,"
                  " poicategory.label, poicategory.description, "
                  " poicategory.enabled from poi inner join poicategory"
                  " on poi.categoryId = poicategory.categoryId");

    if (query.exec()) {
        while (query.next()) {
            qint32 poiId = query.value(0).toInt();
            double latitude = query.value(1).toDouble();
            double longitude = query.value(2).toDouble();
            QString poiLabel = query.value(3).toString();
            QString poiDescription = query.value(4).toString();
            qint32 categoryId = query.value(5).toInt();
            QString categoryLabel = query.value(6).toString();
            QString categoryDescription = query.value(7).toString();
            qint32 enabled = query.value(8).toInt();

            PoiCategory category;
            category.setCategoryId(categoryId);
            category.setLabel(categoryLabel);
            category.setDescription(categoryDescription);
            category.setEnabled(enabled);

            Poi poi;
            poi.setPoiId(poiId);
            poi.setLatitude(latitude);
            poi.setLongitude(longitude);
            poi.setLabel(poiLabel);
            poi.setDescription(poiDescription);
            poi.setCategory(category);

            pois.append(poi);
        }

        return true;
    }

    return false;
}

//! Gets POIs by key word to vector from database.
bool PoiDatabase::getPoisByKeyWord(QVector<Poi> & pois, QString keyWord)
{
    const QString wildcard("%");
    QSqlQuery query(db);

    query.prepare("select poi.poiId, poi.latitude, poi.longitude, "
                  "poi.label, poi.description, poi.categoryId, "
                  "poicategory.label, poicategory.description, "
                  "poicategory.enabled from poi inner join poicategory "
                  "on poi.categoryId = poicategory.categoryId "
                  "where poi.label like ? or poi.description like ? "
                  "order by poi.label desc");
    QString value = QString(wildcard).append(keyWord).append(wildcard);
    query.bindValue(0, value);
    query.bindValue(1, value);

    if (query.exec()) {
        while (query.next()) {
            qint32 poiId = query.value(0).toInt();
            double latitude = query.value(1).toDouble();
            double longitude = query.value(2).toDouble();
            QString poiLabel = query.value(3).toString();
            QString poiDescription = query.value(4).toString();
            qint32 categoryId = query.value(5).toInt();
            QString categoryLabel = query.value(6).toString();
            QString categoryDescription = query.value(7).toString();
            qint32 enabled = query.value(8).toInt();

            PoiCategory category;
            category.setCategoryId(categoryId);
            category.setLabel(categoryLabel);
            category.setDescription(categoryDescription);
            category.setEnabled(enabled);

            Poi poi;
            poi.setPoiId(poiId);
            poi.setLatitude(latitude);
            poi.setLongitude(longitude);
            poi.setLabel(poiLabel);
            poi.setDescription(poiDescription);
            poi.setCategory(category);

            pois.append(poi);
        }

        return true;
    }

    return false;
}

//! Inserts some default POI categories to database.
void PoiDatabase::insertDefaultCategories()
{
    QSqlQuery query(db);
    query.prepare("insert into poicategory (label, description, enabled) "
                  "VALUES (?, ?, 1); ");
    query.bindValue(0, "Service Station");
    query.bindValue(1, "Stations for purchasing fuel for vehicles.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Restaurant");
    query.bindValue(1, "Places to eat or drink.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Residence");
    query.bindValue(1, "Houses, apartments, or other residences of import.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Shopping/Services");
    query.bindValue(1, "Places to shop or acquire services.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Recreation");
    query.bindValue(1, "Indoor or Outdoor places to have fun.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Transportation");
    query.bindValue(1, "Bus stops, airports, train stations, etc.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Lodging");
    query.bindValue(1, "Places to stay temporarily or for the night.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "School");
    query.bindValue(1, "Elementary schools, college campuses, etc.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Business");
    query.bindValue(1, "General places of business.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Landmark");
    query.bindValue(1, "General landmarks.");
    query.exec();
    query.prepare("insert into poicategory (label, description, enabled) "
                  "values (?, ?, 1); ");
    query.bindValue(0, "Other");
    query.bindValue(1, "Miscellaneous category for everything else.");
    query.exec();
}

//! Destructor.
PoiDatabase::~PoiDatabase()
{
}
