/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "poicategoriesdialog.h"
#include "editcategorydialog.h"
#include "addcategorydialog.h"
#include "poicategory.h"
#include "poidatabase.h"
#include <QAbstractItemView>
#include <QDebug>
#include <QMessageBox>
#include <QStandardItem>
#include <QStandardItemModel>
#include <QString>

//! Constructor.
PoiCategoriesDialog::PoiCategoriesDialog(QtMapper &mainWindow,
                                         QWidget *parent) :
    QDialog(parent), 
    column1Width(150),
    column2Width(250),
    mainWindow(mainWindow)
{
    setupUi(this);
    connect(pushButtonOk, SIGNAL(clicked()), this, SLOT(accept()));
    connect(pushButtonEdit, SIGNAL(clicked()),
            this, SLOT(showEditCategoryDialog()));
    connect(pushButtonAdd, SIGNAL(clicked()),
            this, SLOT(showAddCategoryDialog()));
    connect(pushButtonDelete, SIGNAL(clicked()),
            this, SLOT(showDeleteCategoryDialog()));

	poiDb = mainWindow.getPoiDb();
    
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
	poiDb->getPoiCategories(categories);
	setData();
}

//! Sets the data of POI categories to view.
void PoiCategoriesDialog::setData()
{
    QStandardItem* item1 = new QStandardItem(QString(tr("Label")));
    QStandardItem* item2 = new QStandardItem(QString(tr("Description")));

    QStandardItemModel* model = new QStandardItemModel(this);
 
    model->setHorizontalHeaderItem(0, item1);
    model->setHorizontalHeaderItem(1, item2);

    qint32 i = 0;

    for (QVector<PoiCategory>::iterator categoryIterator = categories.begin();
            categoryIterator != categories.end(); ++categoryIterator) {
        
        QStandardItem* item1 = new QStandardItem(QString(categoryIterator
                                                         ->getLabel()));
        QStandardItem* item2 = new QStandardItem(QString(categoryIterator
                                                        ->getDescription()));
        item1->setEditable(false);
        item2->setEditable(false);
     
        model->setItem(i, 0, item1);
        model->setItem(i, 1, item2);
        
        i++;
    }

    tableView->setModel(model);
    tableView->setSelectionMode(QAbstractItemView::SingleSelection);
    tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
    tableView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    tableView->setColumnWidth(0, column1Width);
    tableView->setColumnWidth(1, column2Width);
  
}
//!* Shows dialog Edit Category and handles updating of database when
//! POI category data is changed.
void PoiCategoriesDialog::showEditCategoryDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    EditCategoryDialog ec(this);
	qint32 i = tableView->currentIndex().row();
		
	if (i < 0 || i > categories.count()) {
		i = 0;
	}
	
	if (categories.count() == 0) {
		return;
	}
	
	QVector<PoiCategory>::iterator currentItem = &categories[i];
	
	ec.lineEditLabel->setText(currentItem->getLabel());
	ec.textEditDescription->setText(currentItem->getDescription());
		
	if (ec.exec()==QDialog::Accepted) {
				
		currentItem->setLabel(ec.lineEditLabel->text());
		currentItem->setDescription(ec.textEditDescription->toPlainText());
	    currentItem->setEnabled(1);
						
		poiDb->updatePoiCategory(*currentItem);
		setData();

        tableView->selectRow(i);
    }
}

//! Shows dialog Add Category and handles inserting data to database when
//! POI category is added.
void PoiCategoriesDialog::showAddCategoryDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    AddCategoryDialog ac(this);
		
	if (ac.exec()==QDialog::Accepted) {
		
		PoiCategory category;
		category.setLabel(ac.lineEditLabel->text());
		category.setDescription(ac.textEditDescription->toPlainText());
		category.setEnabled(1);
				
		QVector<PoiCategory> allCategories;
		bool similar = false;
		
		poiDb->getPoiCategories(allCategories);
				
		for (qint32 i = 0; i < allCategories.count(); i++ ) {
            
			if (category.isSimilar(allCategories.at(i))) {
				similar = true;
			}
		}
		
		if (similar == true) {
            QMessageBox mb(this);
			mb.setStandardButtons(QMessageBox::Ok);
            mb.setWindowTitle(tr("Error"));
            QString text(tr("POI category with that name already exists.\n"
                            "No new category created."));
			mb.setText(text);
			mb.exec();
		} else {
			categories.clear();
	      	poiDb->insertPoiCategory(category);
	        poiDb->getPoiCategories(categories);
			setData();
		}
   	}
	    
}
//! Shows dialog Delete Category and handles removing data from database
//! when POI category is deleted.
void PoiCategoriesDialog::showDeleteCategoryDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }    
    
    QMessageBox mb(this);

    qint32 i = tableView->currentIndex().row();

    if (i < 0 || i > categories.count()) {
        i = 0;
    }

    if (categories.count() == 0) {
        return;
    }

    QVector<PoiCategory>::const_iterator currentItem = &categories.at(i);

    mb.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
    mb.setWindowTitle(tr("Confirm deletion"));
    mb.setText(QString(tr("Delete category \"%1\"?\n"
                          "WARNING: All POIs in category will be deleted.")
                          .arg(currentItem->getLabel())));

    if (mb.exec() == QMessageBox::Ok) {

		poiDb->removePoiCategory(currentItem->getCategoryId());
        poiDb->removePoiByCategoryId(currentItem->getCategoryId());
        categories.clear();
        poiDb->getPoiCategories(categories);
		setData();
	}
}

//! Destructor.
PoiCategoriesDialog::~PoiCategoriesDialog()
{

}
