/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "plot.h"
#include "unitconverter.h"
#include "definitions.h"
#include <QPainter>
#include <QDebug>
#include <QRect>
#include <QRadialGradient>
#include <QPointF>
#include "math.h"

//! Default constructor
Plot::Plot()
{
    init();
}

//! Constructor which takes starting measures
Plot::Plot(int x, int y, int width, int height)
{
    this->x = x;
    this->y = y;
    this->width = width;
    this->height = height;
    init();
}

//! Constructor which takes starting measures as QRect
Plot::Plot(QRect &rect)
{
    this->x = rect.x();
    this->y = rect.y();
    this->width = rect.width();
    this->height = rect.height();
    init();
}

//! Initializes variables. Called from all constructors.
void Plot::init()
{
    if (height > width) {
        sideLength = width;
    } 
    else if (height < width) {
        sideLength = height;
    }
    
    this->circle = QPen(Qt::gray, 1, Qt::SolidLine);
    radiusLines = QPen(Qt::gray, 1, Qt::SolidLine);
    drawRect = new QRect((int)centerXpos, y, sideLength, sideLength);
}

//! Updates measures with the given parameters.
void Plot::updateArea(QRect &rect)
{
    this->x = rect.x();
    this->y = rect.y();
    this->width = rect.width();
    this->height = rect.height();
    init();
}

//! return draw area by reference wrapped in QRect.
QRect &Plot::getDrawArea()
{
    drawRect->setX((int)centerXpos);
    drawRect->setY(y);
    drawRect->setBottom(sideLength);
    drawRect->setWidth(sideLength);
    return *drawRect;
}

//! Called by paintevent. Draws background for the satellites'positions.
void Plot::update(QPainter *painter)
{
    UnitConverter *uc = new UnitConverter();
    painter->setPen(this->circle);

    qreal centerY = y + sideLength / 2;
    centerXpos = x + width / 2 - sideLength / 2;
    qreal realCenterX = x + width / 2;

    // Line spacing is 30 degrees.
    qreal degIncreasement = 30;
    qreal degrees = 30;

    QRadialGradient gradient(realCenterX, centerY, sideLength,
                             realCenterX, centerY);
    gradient.setColorAt(0.2, Qt::white);
    gradient.setColorAt(1, Qt::blue);
    painter->setBrush(gradient);

    painter->drawEllipse((int)centerXpos, y, sideLength, sideLength);
    painter->drawEllipse((int)(centerXpos + sideLength / 4), y + sideLength / 4,
                         sideLength / 2, sideLength / 2);
    painter->setPen(radiusLines);

    for (int i = 0; i < maxSatellitesCount; i++) {
        QPointF startPoint(realCenterX, centerY) ;
        QPointF endPoint(realCenterX + (sin(uc->deg2rad(degrees))*sideLength / 2),
                                    centerY + (cos(uc->deg2rad(degrees))*sideLength / 2)) ;
        painter->drawLine(startPoint,endPoint);
        degrees += degIncreasement;
    }

    delete uc ;
}
