/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __MAPVIEW_H__
#define __MAPVIEW_H__

#include <QGraphicsView>
#include <QList>
#include <QPixmap>
#include <QPointer>
#include <QVector>
#include "definitions.h"
#include "poiinfo.h"
#include "trackpoint.h"

class LonLatBox;
class MapScene;
class MapTile;
class NetworkActivityIndicator;
class RouteData;
class DisplayBox;
class QResizeEvent;
class QuickMenubar;
class SpeedDisplay ;
class Waypoint;
/*!
 * MapView is a scrollable area to display the map images in.
 * The tiles are added in rows and columns so there can be only
 * a certain number of map tiles on screen at once.
 * Inherits QGraphicsView.
 */
class MapView : public QGraphicsView
{
    Q_OBJECT

public:
    /*! Constructor.
     *
     * \param parent The parent widget. Default is 0 (no parent).
     */
    MapView(QWidget *parent = 0);

    //! Destructor.
    ~MapView();

    /*!
     * \brief Enables and disables scrolling/dragging the view
     *
     * When current gps position is locked center on sreen via this function
     * is map dragging enabled or disabled.
     * \param locked If true dragging is not allowed and false allows dragging.
     */
    void lockView(bool locked) ;

    /*!
     * Add a map tile in the view.
     *
     * Adds a map tile in the view at the specified location.
     * The location is calculated by multiplying the given row
     * and column values with the width and height of the tile.
     *
     * \param tile The tile to add.
     * \param row The row to add the tile in.
     * \param column The column to add the tile in.
     */
    void addMapTile(MapTile *tile, qint32 row, qint32 column);

    //! Remove all items from the view.
    void clear();

    /*!
     * Draw the GPS location marker at the specified coordinates.
     *
     * \param x The x coordinate to draw the marker.
     * \param y The y coordinate to draw the marker.
     * \param zoom The current zoom level of the map.
     */
    void drawLocationMarker(qint32 x, qint32 y, qint32 zoom);

    /*!
    * Shows location or POI info text and marker.
    *
    * \param x The x coordinate to draw the marker.
    * \param y The y coordinate to draw the marker.
    * \param zoom The current zoom level of the map.
    * \param infoText Info text of location info or POI info
    * \param infoType Value whether info is location info or POI info.
    */
    void showLocationInfo(qint32 x, qint32 y, qint32 zoom,
                          QString &infoText, InfoType infoType);

    //! Returns the scene property.
    MapScene *getScene() const {
        return scene;
    }

    //! Returns the current position marker
    QGraphicsPixmapItem *getMarker();

    //! Removes the current position marker if it exists, and updates the view
    void removeGpsMarker();

    //! Draws the quick menu
    void drawQuickMenu();

    /*!
     * \brief Set the current route information.
     *
     * A pointer to the route information object,
     * which is owned by the QtMapper class.
     *
     * \param data The route information.
     */
    void setRouteData(RouteData *data);

    /*!
     * Show or hide the route display.
     *
     * \param enable True shows the route display, false hides it.
     */
    void enableRouteDisplay(bool enable);

    /*!
     * Shows or hides the track.
     *
     * \param enable True shows the track, false hides it.
     */
    void enableTrack(bool enable);

    /*!
     * Shows or hides the quick menu.
     *
     * \param enable True shows the quick menu, false hides it.
     */
    void enableQuickMenu(bool enable);

    //! Returns the routeDisplayVisible property.
    bool routeDisplayEnabled() const {
        return routeDisplayVisible;
    }

    //! Returns the track visible property.
    bool trackEnabled() const {
        return trackVisible;
    }

    //! Returns the routeLineColor property.
    QColor getRouteLineColor() const {
        return routeLineColor;
    }

    //! Sets the routeLineColor property.
    void setRouteLineColor(QColor color) {
        routeLineColor = color;
    }

    //! Returns the route line color property.
    int getRouteLineThickness() const {
        return routeLineThickness;
    }

    //! Sets the routeLineThickness property.
    void setRouteLineThickness(int thickness) {
        routeLineThickness = thickness;
    }

    /*!
     * Draws a line along the route (used also when drawing track).
     *
     * \param path The path to draw.
     */
    void drawRouteLine(const QPainterPath &path);

    //! Draws distance info of track.
    void drawTrackDistance();

    /*!
     * Stores track data to vector.
     *
     * \param trackPoints Contains the data of a track.
     */
    void setTrackData(QVector<TrackPoint> trackPoints) {
        this->trackPoints = trackPoints;
    }

    /*!
     * Selects the next waypoint along the route.
     *
     * \return The index of the next waypoint in the RouteData object.
     */
    int nextWaypoint();

    /*!
     * Get the position of the last waypoint in the route information.
     * \return The last waypoint index.
     */
    int getRouteEnd() {
        return routeEnd;
    }

    /*!
     * Get the position of the current waypoint in the route information.
     * \return The current waypoint index.
     */
    int getCurrentWaypoint() {
        return currentWaypoint;
    }

    /*!
     * Sets tracking button to "ON-mode" on the QuickMenuBar.
     * If users selects tracking on from the drop-down menu this is called to
     * indicate tracking ON.
     * \param mode True means tracking on and false tracking is off.
     */
    void setTrackingEnabled(bool mode);

    //! Shows and hides the box with latitude and longitude information
    void toggleLonLatBox(bool visible);

    /*!
     * Show or hide the network activity indicator.
     *
     * \param show True shows the indicator, false hides it.
     */
    void showNetworkActivityIndicator(bool show);

    //! Get the network indicator.
    NetworkActivityIndicator *getNetworkActivityIndicator() const
    { return networkActivityIndicator; }

    //! Sets longitude and latitude values to LonLatBox
    /*!
     * \param longitude New value for longitude
     * \param latitude New value for latitude
     */
    void setLonLatBoxValues(qreal longitude, qreal latitude);

    //! Returns true if cursor is inside the quickmenu's area
    bool isCursorInQuickMenu();

    /*!
     * sets new speed value for the speed display.
     */
    void setNewSpeed(qreal speed) ;

private:
    //! True if cursor is inside the quickmenu's area
    bool cursorInQuickMenu;

    //! Object for the speed display class.
    SpeedDisplay *speedDisplay ;

    //!Updates speed display's position when updating view.
    void updateSpeedDisplayPos() ;

    //! A box that displays longitude and latitude information
    QPointer<LonLatBox> lonLatBox;

    /*!
     * Enabling and disabling speed display on the screen.
     */
    void drawSpeedDisplay() ;

    /*!
     * \brief When the view is locked, the used drag mode is stored here.
     *
     * And when the view is again unlocked, the drag mode is set to whatever it
     * was before locking, by using what's stored here
     */
    QGraphicsView::DragMode previousDragMode;

    //! Whether screen is locked on current GPS position or not
    bool locked ;

    //! Whether tracking is in use
    bool trackingEnabled;

    //! The GPS location marker image resource
    const QString markerImageResource;
    
    //! The location marker image resource
    const QString locationInfoImageResource;
    
    //! The POI info image resource
    const QString poiInfoImageResource;
    
    //! For POI info
    const int pixelsUpper;
    
    //! POI info text size
    const int poiTextSize;
    
    //! Location info text size
    const int locationTextSize;
    
    //! For POI info
    const int rectPixelsUpper;
    
    //! POI info rectangle width
    const int rectWidth;
    
    //! For rectangle height of POI info
    const int rectHeightPixelsMore;
    
    //! POI info text width
    const int infoTextWidth;
    
    //! For POI info item
    const int infoItemPixelsUpper;
    
    //! For POI info item
    const int infoItemPixelsRight;
    
    //! Font
    const QString sansSerifFont;
    
    //! Font size
    const int fontSize10pt;
    
    //! Font size
    const int fontSize12pt;
    
    //! Font size
    const int fontSize14pt;
    
    //! For route display
    const QString htmlCenterTagOpen;
    
    //! For route display
    const QString htmlCenterTagClose;
    
    //! Height of track display
    const qint32 trackDisplayHeight;
    
    //! Z value of track display
    const qint32 trackDisplayZValue;
    
    //! Z value of track text
    const qint32 trackTextZValue;
    
    //! Z value of quick menu bar
    const int quickMenubarZValue;
    
    //! Z value of route display
    const int routeDisplayZValue;
    
    //! Z value of route text
    const int routeTextZValue;
    
    //! Z value of route line
    const int routeLineZValue;
    
    //! Height of route display
    const int routeDisplayHeight;
    
    //! Z value of POI info marker
    const int poiInfoMarkerZValue;
    
    //! Z value of info item
    const int infoItemZValue;

    //! Z value of info
    const int infoZValue;

    //! Z value of info label
    const int infoLabelZValue;

    //! Z valyue of the network indicator
    const int networkIndicatorZValue;

    //! The interval between animation frames (ms).
    const int animationTimerInterval;

    //! Z value for the Speed Display
    const int speedDisplayZvalue ;

    //! Starting tag of html parahraph
    const QString startHtmlParagraph;

    //! Starting tag of bold html text
    const QString htmlBoldStart;

    //! Ending tag of bold html text
    const QString htmlBoldEnd;

    //! New line html tag
    const QString htmlNewLine;

    //! Ending tag of html paragraph
    const QString endHtmlParagraph;

    //! Draws the route distances.
    void drawRouteDistances();

    //! Sets the last waypoint of the route.
    void setLastWaypoint();

    //! True if quickmenu is enabled
    bool quickMenuEnabled;

    //! Whether LonLatBox is in use
    bool lonLatBoxInUse;

    //! Width of the LonLatBox
    const int lonLatBoxWidth;

    //! Height of the LonLatBox
    const int lonLatBoxHeight;

    //! Enabling and disabling speed display.
    bool speedDisplayEnabled ;

    // These are for testing
    QPixmap *quickMenuPixmap;

    //QGraphicsPixmapItem *quickMenuGraphicsItem;
    QuickMenubar *quickMenubar ;

    //! horizontal scrollbar's maximum value
    int horizontalScrollbarMaxValue;

    //! horizontal scrollbar's minimum value
    int horizontalScrollbarMinValue;

    //! vertical scrollbar's maximum value
    int verticalScrollbarMaxValue;

    //! vertical scrollbar's minimum value
    int verticalScrollbarMinValue;

    //! MapScene is where da graphics items at.
    MapScene *scene;

    //! The GPS location marker image.
    QPixmap markerImage;

    //! The searched location marker image
    QPixmap locationInfoImage;

    //! The POI info image
    QPixmap poiInfoImage;

    //! Returns a pointer to current location marker
    QGraphicsPixmapItem *marker;

    //! Returns a pointer to searched location marker
    QGraphicsPixmapItem *locationInfoMarker;

    //! Returns a pointer to POI info marker
    QGraphicsPixmapItem *poiInfoMarker;

    //! Container for POI info
    QVector<PoiInfo *> poiInfo;

    //! Container for track points
    QVector<TrackPoint> trackPoints;

    //! Contains value whether the track is visible or not.
    bool trackVisible;

    //! Used to determine whether the route display should be visible.
    bool routeDisplayVisible;

    //! For displaying route information.
    DisplayBox *routeDisplay;

    //! Information about the current route.
    RouteData *currentRouteData;

    //! Used to index the waypoint list in the RouteData object.
    int currentWaypoint;

    //! The last waypoint of a route.
    int routeEnd;

    //! The color used to draw the route line.
    QColor routeLineColor;

    //! The thickness of the route line in pixels.
    int routeLineThickness;

    //! Distance of track.
    qreal distanceOfTrack;

    //! For displaying track information.
    DisplayBox *trackDisplay;

    //! Displayed when there is network activity.
    NetworkActivityIndicator *networkActivityIndicator;

    //! Used to animate objects on the scene.
    QTimer *animationTimer;

    //! Reset the network indicator position.
    void resetNetworkActivityIndicatorPosition();

    //! A list of the map tiles added to the scene.
    QList<MapTile *> mapTilesInScene;

private slots:
    /*!
     * \brief Mapview was scrolled horizontally.
     *
     * Whenever mapview is scrolled horizontally, this slot is called. If the
     * scrollbar's current value is at it's maximum, a mapSceneAtRight signal
     * is emitted. If the value is at it's minimum, mapSceneAtLeft signal is
     * emitted.
     *
     * \param newValue Scrollbar's current value.
     */
    void scrolledHorizontally(int newValue);

    /*!
     * \brief Mapview was scrolled vertically.
     *
     * Whenever mapview is scrolled vertically, this slot is called. If the
     * scrollbar's current value is at it's maximum, a mapSceneAtBottom signal
     * is emitted. If the value is at it's minimum, mapSceneAtTop signal is
     * emitted.
     *
     * \param newValue Scrollbar's current value.
     */
    void scrolledVertically(int newValue);

    /*!
     * Called from wuickmenu when poi is selected by pressing a map.
     * \param x is mouses x-position according to scene.
     * \param y is mouses y-position according to scene.
     */
    void poiSelection(int x, int y);

    /*!
     * Called from quickmenu when zoom in is selected.
     */
    void zoomMapsIn();
    
    /*!
     * Called from quickmenu when zoom out is selected.
     */
    void zoomMapsOut();

    /*!
     * Called from quickmenu when tracking is turned on or off.
     * \param track Describes whether tracking  is on or off.
     */
    void setTracking(bool track) ;

    /*!
     * Called when mouse pressed, used for showing poi information
     *
     * \param e Information about the mouse event.
     */
    void mousePressed(QGraphicsSceneMouseEvent *e);

    /*!
     * Called from scene when mouse is moving
     * \param e Information about the mouse event.
     */
    void mouseMoved(QGraphicsSceneMouseEvent *e);

    //! Draw the route display.
    void drawRouteDisplay();

signals:
    //! Emitted when vertical scrollbar is at top
    void mapSceneAtTop();

    //! Emitted when vertical scrollbar is at bottom
    void mapSceneAtBottom();

    //! Emitted when horizontal scrollbar is at right
    void mapSceneAtRight();

    //! Emitted when horizontal scrollbar is at left
    void mapSceneAtLeft();

    /*!
     * Emitted when tracking is enabled or disabled.
     *
     * \param True if enabled, otherwise false.
     */
    void enableTracking(bool) ;

    //! emitted when zoom out is triggered.
    void zoomedOut() ;

    //! emitted when zoom in is triggered.
    void zoomedIn() ;

    //! emitted when poi is selected.
    void poiSelected(int, int) ;

    //! Emitted when the route has reached the last waypoint.
    void lastWaypointReached();

public slots:
    /*!
     * Called when user checks button in view -dropdown menu.
     * \param True if speed display is enabled and shown on the sreen and false removes it from view.
     */
    void enableSpeedDisplay(bool enabled) ;
};

#endif 
