/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QGraphicsItem>
#include <QGraphicsTextItem>
#include <QGraphicsRectItem>
#include <QGraphicsSceneMouseEvent>
#include <QPixmap>
#include <QPoint>
#include <QScrollBar>
#include <QString>
#include <QTimer>
#include <cmath>

#include "QtMapper.h"
#include "definitions.h"
#include "linepoint.h"
#include "mapscene.h"
#include "maptile.h"
#include "mapview.h"
#include "networkactivityindicator.h"
#include "lonlatbox.h"
#include "quickmenubar.h"
#include "routedata.h"
#include "displaybox.h"
#include "speeddisplay.h"

//! Constructor.
MapView::MapView(QWidget *parent) :
    QGraphicsView(parent),
    markerImageResource(":/resources/images/marker.png"),
    locationInfoImageResource(":/resources/images/goto.png"),
    poiInfoImageResource(":/resources/images/poi.png"),
    pixelsUpper(40),
    poiTextSize(10),
    locationTextSize(10),
    rectPixelsUpper(30),
    rectWidth(200),
    rectHeightPixelsMore(20),
    infoTextWidth(180),
    infoItemPixelsUpper(20),
    infoItemPixelsRight(10),
    sansSerifFont("sans-serif"),
    fontSize10pt(10),
    fontSize12pt(12),
    fontSize14pt(14),
    htmlCenterTagOpen("<center>"),
    htmlCenterTagClose("</center>"),
    trackDisplayHeight(50),
    trackDisplayZValue(12),
    trackTextZValue(14),
    quickMenubarZValue(20),
    routeDisplayZValue(10),
    routeTextZValue(11),
    routeLineZValue(9),
    routeDisplayHeight(80),
    poiInfoMarkerZValue(3),
    infoItemZValue(9),
    infoZValue(8),
    infoLabelZValue(7),
    networkIndicatorZValue(20),
    animationTimerInterval(50),
    speedDisplayZvalue(30),
    startHtmlParagraph("<p>"),
    htmlBoldStart("<b>"),
    htmlBoldEnd("</b>"),
    htmlNewLine("<br>"),
    endHtmlParagraph("</p>"),
    lonLatBoxWidth(150),
    lonLatBoxHeight(50),
    speedDisplayEnabled(false)
{
    setDragMode(QGraphicsView::ScrollHandDrag);
    previousDragMode = QGraphicsView::ScrollHandDrag;
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setFrameStyle(QFrame::Box | QFrame::Plain);
    setViewportUpdateMode(QGraphicsView::FullViewportUpdate);

    setResizeAnchor(QGraphicsView::AnchorViewCenter);

    setMaximumSize(mapTileWidth * (mapTileColumns - 1),
                   mapTileHeight * (mapTileRows - 1));

    scene = new MapScene(this);
    setScene(scene);

    markerImage.load(markerImageResource);
    marker = NULL;

    locationInfoImage.load(locationInfoImageResource);
    locationInfoMarker = NULL;

    poiInfoMarker = NULL;
    poiInfoImage.load(poiInfoImageResource);

    horizontalScrollbarMaxValue = horizontalScrollBar()->maximum();
    horizontalScrollbarMinValue = horizontalScrollBar()->minimum();

    verticalScrollbarMaxValue = verticalScrollBar()->maximum();
    verticalScrollbarMinValue = verticalScrollBar()->minimum();

    connect(horizontalScrollBar(), SIGNAL(valueChanged(int)),
            this, SLOT(scrolledHorizontally(int)));
    connect(verticalScrollBar(), SIGNAL(valueChanged(int)),
            this, SLOT(scrolledVertically(int)));
    connect(scene, SIGNAL(mouseMoved(QGraphicsSceneMouseEvent*)),
            this, SLOT(mouseMoved(QGraphicsSceneMouseEvent*))) ;

    enableRouteDisplay(false);

    trackingEnabled = false;
    locked = false;
    enableQuickMenu(true);
    enableTrack(false);

    networkActivityIndicator = new NetworkActivityIndicator();
    animationTimer = new QTimer(this);
    animationTimer->setInterval(animationTimerInterval);
    connect(animationTimer, SIGNAL(timeout()), scene, SLOT(advance()));

    distanceOfTrack = 0.0;

    lonLatBox = NULL;
    lonLatBoxInUse = false;

    cursorInQuickMenu = false;
}

//! Destructor.
MapView::~MapView()
{
    if (quickMenuEnabled) {
        delete quickMenubar;
    }

    delete networkActivityIndicator;
}

//! Enables and disables scrolling/dragging the view
void MapView::lockView(bool locked)
{
    this->locked = locked;

    if (locked == true) {
        previousDragMode = dragMode();
        setDragMode(QGraphicsView::NoDrag);
    }
    else {
        setDragMode(previousDragMode);
    }
}

//! Called from scene when mouse is moving
void MapView::mouseMoved(QGraphicsSceneMouseEvent *e)
{
    if (!locked) {
        if (quickMenuEnabled) {
            if(quickMenubar->boundingRect().contains(e->scenePos())) {
                if (this->dragMode() == ScrollHandDrag) {
                    setDragMode(QGraphicsView::NoDrag);
                    cursorInQuickMenu = true;
                }
            }
            else {
                if (this->dragMode() == NoDrag) {
                    setDragMode(QGraphicsView::ScrollHandDrag);
                    cursorInQuickMenu = false;
                }
            }
        }
    }
    else {
        if (quickMenuEnabled) {
            if(quickMenubar->boundingRect().contains(e->scenePos())) {
                cursorInQuickMenu = true;
            }
            else cursorInQuickMenu = false;
        }
        else cursorInQuickMenu = false;
    }
}

//! Add a map tile in the view.
void MapView::addMapTile(MapTile *tile, qint32 row, qint32 column)
{
    if (!mapTilesInScene.contains(tile)) {
        mapTilesInScene.append(tile);
        scene->addItem(tile);
        tile->setPos(row * (tile->boundingRect().width() - 1),
                     column * (tile->boundingRect().height() - 1));
    }
    else {
        delete tile;
    }
}

//! Remove all items from the view.
void MapView::clear()
{
    qreal displayedLongitude = 0.0;
    qreal displayedLatitude = 0.0;

    if (lonLatBoxInUse) {
        displayedLongitude = lonLatBox->getDisplayedLongitude();
        displayedLatitude = lonLatBox->getDisplayedLatitude();
    }

    for (int i = 0; i < mapTilesInScene.size(); i++) {
        scene->removeItem(mapTilesInScene.at(i));
        delete mapTilesInScene[i];
    }

    mapTilesInScene.clear();

    for (int i = 0; i < scene->items().size(); i++) {
        QGraphicsItem *item = scene->items().at(i);
        NetworkActivityIndicator *n =
                qgraphicsitem_cast<NetworkActivityIndicator *> (item);

        if (n != 0) {
            if (!n->isActive()) {
                scene->removeItem(n);
                delete n;
                n = 0;
                i--;
            }
        }
        else if (item->parentItem() == 0) {
            scene->removeItem(item);
            delete item;
            i--;
        }
    }

    marker = NULL;
    locationInfoMarker = NULL;
    poiInfoMarker = NULL;
    poiInfo.clear();

    if (quickMenuEnabled) {
        drawQuickMenu();
    }

    if (routeDisplayVisible) {
        drawRouteDisplay();
    }

    if (trackVisible) {
        drawTrackDistance();
    }

    if (speedDisplayEnabled) {
        drawSpeedDisplay() ;
    }

    if (lonLatBoxInUse) {
        toggleLonLatBox(true);
        setLonLatBoxValues(displayedLongitude, displayedLatitude);
    }
}

//! Draw the GPS location marker at the specified coordinates.
void MapView::drawLocationMarker(qint32 x, qint32 y, qint32 zoom)
{
    qint32 newX;
    qint32 newY;

    marker = new QGraphicsPixmapItem(markerImage);
    marker->setZValue(1);

    if (zoom == 0) {
        newX = x - markerImage.width() / 2;
        newY = y - markerImage.height() / 2;
    }
    else {
        newX = x + mapTileWidth - markerImage.width() / 2;
        newY = y + mapTileHeight - markerImage.height() / 2;
    }

    scene->addItem(marker);
    marker->moveBy(newX, newY);
}

//! Returns a pointer to current location marker
QGraphicsPixmapItem *MapView::getMarker()
{
    return marker;
}

//! Mapview was scrolled horizontally
void MapView::scrolledHorizontally(int newValue)
{
    if (speedDisplayEnabled) {
        updateSpeedDisplayPos() ;
    }

    int x = horizontalScrollBar()->value();
    int y = verticalScrollBar()->value();

    int quickMenubarWidth = 0;

    if (networkActivityIndicator && networkActivityIndicator->isActive()) {
        resetNetworkActivityIndicatorPosition();
    }

    if (quickMenuEnabled) {
        quickMenubar->setPos(x, y);
        quickMenubarWidth = static_cast<int> (quickMenubar->getWidth());
    }

    if (lonLatBox) {
        lonLatBox->setPos(x + quickMenubarWidth, y);
    }

    if (routeDisplayVisible) {
        routeDisplay->setPos(newValue + (width() - routeDisplay->getWidth()),
                             verticalScrollBar()->value() +
                             (height() - routeDisplay->getHeight()));
    }

    if (trackVisible) {
        trackDisplay->setPos(newValue + (width() - trackDisplay->getWidth()),
                             verticalScrollBar()->value() +
                             (height() - trackDisplay->getHeight()));
    }

    horizontalScrollbarMaxValue = horizontalScrollBar()->maximum();
    horizontalScrollbarMinValue = horizontalScrollBar()->minimum();

    if (newValue == horizontalScrollbarMaxValue) {
        emit mapSceneAtRight();
    }

    else if (newValue == horizontalScrollbarMinValue) {
        emit mapSceneAtLeft();
    }
}

//! Mapview was scrolled vertically
void MapView::scrolledVertically(int newValue)
{
    if (speedDisplayEnabled) {
        updateSpeedDisplayPos() ;
    }
    if (networkActivityIndicator && networkActivityIndicator->isActive()) {
        resetNetworkActivityIndicatorPosition();
    }

    int x = horizontalScrollBar()->value();
    int y = verticalScrollBar()->value();

    int quickMenubarWidth = 0;

    if (quickMenuEnabled) {
        quickMenubar->setPos(x, y);
        quickMenubarWidth = static_cast<int> (quickMenubar->getWidth());
    }

    if (lonLatBox) {
        lonLatBox->setPos(x + quickMenubarWidth, y);
    }

    if (routeDisplayVisible) {
        routeDisplay->setPos(horizontalScrollBar()->value(),
                             newValue + (height() - routeDisplay->getHeight()));
    }

    if (trackVisible) {
        trackDisplay->setPos(horizontalScrollBar()->value(),
                             newValue + (height() - trackDisplay->getHeight()));
    }

    verticalScrollbarMaxValue = verticalScrollBar()->maximum();
    verticalScrollbarMinValue = verticalScrollBar()->minimum();

    if (newValue == verticalScrollbarMaxValue) {
        emit mapSceneAtBottom();
    }

    else if (newValue == verticalScrollbarMinValue) {
        emit mapSceneAtTop();
    }
}

//! Called from wuickmenu when poi is selected by pressing a map.
void MapView::poiSelection(int x, int y)
{
    emit poiSelected(x, y);
}

//! Called from quickmenu when zoom in is selected.
void MapView::zoomMapsIn()
{
    emit zoomedIn() ;
}

//! Called from quickmenu when zoom out is selected.
void MapView::zoomMapsOut()
{
    emit zoomedOut() ;
}

//! Called from quickmenu when tracking is turned on or off.
void MapView::setTracking(bool track)
{
    emit enableTracking(track) ;
}

//! Draw the quick menu
void MapView::drawQuickMenu()
{
    quickMenubar = new QuickMenubar(0, horizontalScrollBar()->value(),
                                    verticalScrollBar()->value(), 60, NULL) ;
    scene->addItem(quickMenubar) ;
    quickMenubar->setZValue(quickMenubarZValue) ;

    connect(quickMenubar, SIGNAL(poiSelected(int, int)),
            this, SLOT(poiSelection(int, int))) ;
    connect(quickMenubar, SIGNAL(zoomIn()), this, SLOT(zoomMapsIn())) ;
    connect(quickMenubar, SIGNAL(zoomOut()), this, SLOT(zoomMapsOut())) ;
    connect(quickMenubar, SIGNAL(tracking(bool)),
            this, SLOT(setTracking(bool)));
    connect(scene, SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
            quickMenubar, SLOT(mousePressed(QGraphicsSceneMouseEvent*))) ;
    connect(scene, SIGNAL(mouseButtonReleased(QGraphicsSceneMouseEvent*)),
            quickMenubar, SLOT(mouseReleased(QGraphicsSceneMouseEvent*))) ;

    if (trackingEnabled) {
        quickMenubar->setTrackingSelected(true);
        quickMenubar->update();
    }
}

//!Shows location info text and marker
void MapView::showLocationInfo(qint32 x, qint32 y, qint32 zoom,
                               QString &infoText, InfoType infoType)
{
    qint32 newX;
    qint32 newY;

    QString labelOfInfoText = infoText.section("\n",0,0);
    QString descriptionOfInfoText = infoText.section("\n",1);
    QString htmlText(startHtmlParagraph);

    htmlText.append(htmlBoldStart);
    htmlText.append(labelOfInfoText);
    htmlText.append(htmlBoldEnd);
    htmlText.append(htmlNewLine);
    htmlText.append(descriptionOfInfoText);
    htmlText.append(endHtmlParagraph);

    QGraphicsTextItem* infoItem = new QGraphicsTextItem();
    infoItem->setHtml(htmlText);

    if (infoType == infoTypePoi) {

        poiInfoMarker = new QGraphicsPixmapItem(poiInfoImage);
        infoItem->setFont(QFont(sansSerifFont, poiTextSize));
        poiInfoMarker->setZValue(poiInfoMarkerZValue);
        infoItem->setZValue(infoItemZValue);

        if (zoom == 0) {
            newX = x - poiInfoImage.width() / 2;
            newY = y - poiInfoImage.height() / 2;
        }
        else {
            newX = x + mapTileWidth - poiInfoImage.width() / 2;
            newY = y + mapTileHeight - poiInfoImage.height() / 2;
        }

        scene->addItem(poiInfoMarker);
        scene->addItem(infoItem);
        poiInfoMarker->moveBy(newX, newY);
        infoItem->setTextWidth(infoTextWidth);
        qint32 rectHeight = (qint32)infoItem->boundingRect().height();

        PoiInfo *info = new PoiInfo(rectWidth,
                                    rectHeight + rectHeightPixelsMore);
        info->setDescription(infoText);
        info->setX((qint32)poiInfoMarker->scenePos().x());
        info->setY((qint32)poiInfoMarker->scenePos().y());
        scene->addItem(info);
        info->moveBy(info->getX(),
                     info->getY() - (rectHeight + rectPixelsUpper));
        info->setZValue(infoZValue);

        infoItem->moveBy(info->getX() + infoItemPixelsRight,
                         info->getY() - (rectHeight + infoItemPixelsUpper));

        info->setToggled(false);
        info->setInfoItem(infoItem);
        info->hide();
        info->getInfoItem()->hide();

        if (zoom > 16) {
            QGraphicsTextItem* infoLabel = new QGraphicsTextItem(labelOfInfoText);
            infoLabel->setFont(QFont(sansSerifFont, poiTextSize));
            scene->addItem(infoLabel);
            info->setLabel(labelOfInfoText);
            infoLabel->moveBy(info->getX() + infoItemPixelsRight, info->getY() - infoItemPixelsUpper);
            infoLabel->setZValue(infoLabelZValue);
            info->setInfoLabel(infoLabel);
            info->getInfoLabel()->show();
        }

        poiInfo.append(info);

    }
    else {

        locationInfoMarker = new QGraphicsPixmapItem(locationInfoImage);
        infoItem->setFont(QFont(sansSerifFont, locationTextSize, QFont::Bold));

        locationInfoMarker->setZValue(5);
        infoItem->setZValue(6);

        if (zoom == 0) {
            newX = x - poiInfoImage.width() / 2;
            newY = y - poiInfoImage.height() / 2;
        }
        else {
            newX = x + mapTileWidth - poiInfoImage.width() / 2;
            newY = y + mapTileHeight - poiInfoImage.height() / 2;
        }

        scene->addItem(locationInfoMarker);
        scene->addItem(infoItem);
        locationInfoMarker->moveBy(newX, newY);
        infoItem->moveBy(newX, newY - pixelsUpper);
    }
}

//Called when mouse pressed, used for showing poi information
void MapView::mousePressed(QGraphicsSceneMouseEvent *e)
{
   for (qint32 i = 0; i < poiInfo.count(); i++) {

        //if poi info is clicked area
        if (e->scenePos().x() < (poiInfo.at(i)->getX() + poiInfoImage.width()) &&
            e->scenePos().x() > poiInfo.at(i)->getX() &&
            e->scenePos().y() < (poiInfo.at(i)->getY() + poiInfoImage.height()) &&
            e->scenePos().y() > poiInfo.at(i)->getY()) {

            //show poi info of selected item and hide others
            if (poiInfo.at(i)->getToggled() == false) {
                poiInfo.at(i)->show();
                poiInfo.at(i)->getInfoItem()->show();
                poiInfo.at(i)->setToggled(true);

                if (poiInfo.at(i)->getInfoLabel()) {
                    poiInfo.at(i)->getInfoLabel()->hide();
                }

                for (qint32 j = 0; j < poiInfo.count(); j++) {

                    if (j != i) {
                        poiInfo.at(j)->hide();
                        poiInfo.at(j)->getInfoItem()->hide();
                        poiInfo.at(j)->setToggled(false);

                        if (poiInfo.at(j)->getInfoLabel()) {
                            poiInfo.at(j)->getInfoLabel()->show();
                        }
                    }
                }

            } //hide poi info
            else {
                poiInfo.at(i)->hide();
                poiInfo.at(i)->getInfoItem()->hide();

                if (poiInfo.at(i)->getInfoLabel()) {
                    poiInfo.at(i)->getInfoLabel()->show();
                }
                poiInfo.at(i)->setToggled(false);
            }
        }
    }
}

//! Set the current route information.
void MapView::setRouteData(RouteData *data)
{
    currentRouteData = data;
    setLastWaypoint();
}

//! Show or hide the route display.
void MapView::enableRouteDisplay(bool enable)
{
    routeDisplayVisible = enable;
    currentWaypoint = 0;

    if (enable) {
        drawRouteDisplay();
    }
}

//! show or hide the track.
void MapView::enableTrack(bool enable)
{
    trackVisible = enable;
}

//! Show or hide the quick menu.
void MapView::enableQuickMenu(bool enable)
{
    quickMenuEnabled = enable;
    if (enable) {
        drawQuickMenu();
    }
    else
    {
        delete quickMenubar ;
    }
}

//! Draw an area in the view where the route information will appear.
void MapView::drawRouteDisplay()
{
    //! Draw the area where the route information will be drawn.
    routeDisplay = new DisplayBox(width(), routeDisplayHeight);
    routeDisplay->setPos(horizontalScrollBar()->value(),
                         verticalScrollBar()->value() +
                         (height() - routeDisplay->getHeight()));
    routeDisplay->setZValue(routeDisplayZValue);
    scene->addItem(routeDisplay);

    // Draw the route name.
    QString routeNameText = QString().append(htmlCenterTagOpen)
                            .append(currentRouteData->getName())
                            .append(htmlCenterTagClose);

    QGraphicsTextItem *routeName = new QGraphicsTextItem(routeDisplay);
    routeName->setFont(QFont(sansSerifFont, fontSize10pt));
    routeName->font().setCapitalization(QFont::Capitalize);
    routeName->setTextWidth(width());
    routeName->setHtml(routeNameText);
    routeName->setZValue(routeTextZValue);

    // Draw the waypoint text.
    QString waypointText = QString().append(htmlCenterTagOpen)
                           .append(currentRouteData->getWaypoints()
                                   .at(currentWaypoint)->getName())
                           .append(htmlCenterTagClose);

    QGraphicsTextItem *waypoint = new QGraphicsTextItem(routeDisplay);
    waypoint->setFont(QFont(sansSerifFont, fontSize14pt, QFont::Bold));
    waypoint->setTextWidth(width());
    waypoint->setHtml(waypointText);
    waypoint->setZValue(routeTextZValue);
    waypoint->moveBy(0, routeDisplayHeight / 2 -
                     waypoint->boundingRect().height() / 2);

    drawRouteDistances();
}

//! Draws the route distances.
void MapView::drawRouteDistances()
{
    QtMapper *mainWindow = qobject_cast<QtMapper *> (parent());

    qreal currentLon = mainWindow->getCurrentGpsLocation().getLongitude();
    qreal currentLat = mainWindow->getCurrentGpsLocation().getLatitude();
    qreal wpLon = currentRouteData->getWaypoints()
                  .at(currentWaypoint)->getLongitude();
    qreal wpLat = currentRouteData->getWaypoints()
                  .at(currentWaypoint)->getLatitude();

    qreal distanceToWp = mainWindow->distanceBetweenCoordinates(currentLat,
                         currentLon,
                         wpLat,
                         wpLon);

    QString distanceText = QString().append(htmlCenterTagOpen)
                           .append(tr("Next waypoint "))
                           .append(QString("%1").arg(distanceToWp, 0, 'f', 2))
                           .append(tr(" km"))
                           .append(tr("\tTotal "))
                           .append(currentRouteData->getTotalDistance())
                           .append(htmlCenterTagClose);

    QGraphicsTextItem *distance = new QGraphicsTextItem(routeDisplay);
    distance->setFont(QFont(sansSerifFont, fontSize12pt, QFont::Bold));
    distance->setTextWidth(width());
    distance->setHtml(distanceText);
    distance->setZValue(routeTextZValue);
    distance->moveBy(0, routeDisplayHeight - distance->boundingRect().height());
}

//! Draws a line along the route (used also when drawing track).
void MapView::drawRouteLine(const QPainterPath &path)
{
    QGraphicsPathItem *route = scene->addPath(path, QPen(routeLineColor,
                               routeLineThickness));
    route->setZValue(routeLineZValue);
}

//! Selects the next waypoint along the route.
int MapView::nextWaypoint()
{
    currentWaypoint++;
    return currentWaypoint;
}

//! Sets the last waypoint of the route.
void MapView::setLastWaypoint()
{
    routeEnd = currentRouteData->getWaypoints().size() - 1;
}

//! Draws distance info of track.
void MapView::drawTrackDistance()
{
    QtMapper *mainWindow = qobject_cast<QtMapper *> (parent());
    qreal distance = 0.0;

    for (qint32 i = 0; i + 1 < trackPoints.count(); i++) {

        distance = mainWindow
                   ->distanceBetweenCoordinates(trackPoints[i].getLatitude(),
                                                trackPoints[i].getLongitude(),
                                                trackPoints[i+1].getLatitude(),
                                                trackPoints[i+1].getLongitude());
        distanceOfTrack += distance;

    }

    QString modifiedDistance("Distance : ");
    modifiedDistance.append(QString("%1").arg(distanceOfTrack, 0, 'f', 3));
    modifiedDistance.append(tr(" km"));

    //! Draw the area where the track information will be drawn.
    trackDisplay = new DisplayBox(width(), trackDisplayHeight);
    trackDisplay->setPos(horizontalScrollBar()->value(),
                         verticalScrollBar()->value() +
                         (height() - trackDisplay->getHeight()));
    trackDisplay->setZValue(trackDisplayZValue);
    scene->addItem(trackDisplay);

    QGraphicsTextItem *trackDistance = new QGraphicsTextItem(trackDisplay);
    trackDistance->setFont(QFont(sansSerifFont, fontSize14pt));
    trackDistance->setPlainText(modifiedDistance);
    trackDistance->setZValue(trackTextZValue);
    trackDistance->moveBy(0, trackDisplayHeight / 2 -
                          trackDistance->boundingRect().height() / 2);

    distanceOfTrack = 0.0;
}

//! Called from quickmenu when tracking is turned on or off.
void MapView::setTrackingEnabled(bool mode)
{
    trackingEnabled = mode;
    quickMenubar->setTrackingSelected(mode);
    quickMenubar->update();
}

//! Removes the current position marker if it exists, and updates the view
void MapView::removeGpsMarker()
{
    if (marker != NULL) {
        delete marker;
        marker = NULL;
        update();
    }
}

//! Shows and hides the box with latitude and longitude information
void MapView::toggleLonLatBox(bool visible)
{
    lonLatBoxInUse = visible;

    if (visible) {
        lonLatBox = new LonLatBox(lonLatBoxWidth, lonLatBoxHeight);
        lonLatBox->setZValue(quickMenubarZValue);
        scene->addItem(lonLatBox);
        if (quickMenuEnabled) {
            lonLatBox->setPos(horizontalScrollBar()->value()
                                  + quickMenubar->getWidth(),
                              verticalScrollBar()->value());
        }
        else {
            lonLatBox->setPos(horizontalScrollBar()->value(),
                              verticalScrollBar()->value());
        }
    }
    else {
        scene->removeItem(lonLatBox);
        delete lonLatBox;
        lonLatBox = NULL;
    }
}

//! Show or hide the network activity indicator.
void MapView::showNetworkActivityIndicator(bool show)
{
    if (show && !networkActivityIndicator->isActive()) {
        networkActivityIndicator->setZValue(networkIndicatorZValue);
        networkActivityIndicator->setActive(true);
        scene->addItem(networkActivityIndicator);
        resetNetworkActivityIndicatorPosition();
        animationTimer->start();
    }
    else if (networkActivityIndicator->isActive()) {
        networkActivityIndicator->setActive(false);
        scene->removeItem(networkActivityIndicator);
        animationTimer->stop();
    }
}

//! Reset the network indicator position.
void MapView::resetNetworkActivityIndicatorPosition()
{
    qreal x = horizontalScrollBar()->value() + viewport()->width() -
              networkActivityIndicator->boundingRect().width();
    qreal y = verticalScrollBar()->value();

    if (speedDisplayEnabled) {
        y += speedDisplay->boundingRect().height();
    }

    networkActivityIndicator->setPos(x, y);
}

//! create or delete speed display.
void MapView::drawSpeedDisplay()
{
    if (speedDisplayEnabled)
    {
        speedDisplay = new SpeedDisplay() ;
        speedDisplay->setPos(horizontalScrollBar()->value() + viewport()->width() -
                  speedDisplay->boundingRect().width(), verticalScrollBar()->value()) ;
        scene->addItem(speedDisplay) ;
        speedDisplay->setZValue(speedDisplayZvalue) ;
    }
    else
    {
        delete speedDisplay ;
    }
}

//! when place in the scene changes this is called to reorder place.
void MapView::updateSpeedDisplayPos()
{
    if (speedDisplayEnabled)
    {
        qreal x = horizontalScrollBar()->value() + viewport()->width() -
              speedDisplay->boundingRect().width();
        qreal y = verticalScrollBar()->value();
        speedDisplay->setPos(x, y);
    }
}

//! Sets longitude and latitude values to LonLatBox
void MapView::setLonLatBoxValues(qreal longitude, qreal latitude)
{
    if (lonLatBox) {
        lonLatBox->setCoordinateValues(longitude, latitude);
    }
}

//! Returns true if cursor is inside the quickmenu's area
bool MapView::isCursorInQuickMenu()
{
    return cursorInQuickMenu;
}

//! enabling and disabling speed display via this function.
void MapView::enableSpeedDisplay(bool enabled)
{
    speedDisplayEnabled = enabled;
    drawSpeedDisplay();
    resetNetworkActivityIndicatorPosition();
}

//! set new speed for the speed display.
void MapView::setNewSpeed(qreal speed)
{
    if (speedDisplayEnabled) {
        speedDisplay->updateSpeed(speed) ;
    }
}
