/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "maptiledownloaddialog.h"
#include <QPushButton>
#include <QDoubleValidator>

//! Constructor
MapTileDownloadDialog::MapTileDownloadDialog(int minZoomLevel,
                                             int maxZoomLevel,
                                             QWidget *parent) :
    QDialog(parent),
    minimumZoom(minZoomLevel),
    maximumZoom(maxZoomLevel),
    minimumLatitude(-85.05113),
    maximumLatitude(85.05113),
    decimals(5),
    minimumLongitude(-180),
    maximumLongitude(180)
{
    setupUi(this);
    buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);

    connect(buttonBox, SIGNAL(accepted()), this, SIGNAL(okButtonClicked()));
    connect(buttonBox, SIGNAL(rejected()), this, SLOT(close()));

    zoomLevelSpinBox->setMinimum(minimumZoom);
    zoomLevelSpinBox->setMaximum(maximumZoom);

    QDoubleValidator *latitudeValidator = new QDoubleValidator(
                              minimumLatitude, maximumLatitude, decimals, this);

    fromLatitudeLineEdit->setValidator(latitudeValidator);
    toLatitudeLineEdit->setValidator(latitudeValidator);

    QDoubleValidator *longitudeValidator = new QDoubleValidator(
                            minimumLongitude, maximumLongitude, decimals, this);

    fromLongitudeLineEdit->setValidator(longitudeValidator);
    toLongitudeLineEdit->setValidator(longitudeValidator);


    connect(fromLatitudeLineEdit, SIGNAL(textChanged(const QString &)),
            this, SLOT(checkLineEditInputs()));
    connect(toLatitudeLineEdit, SIGNAL(textChanged(const QString &)),
            this, SLOT(checkLineEditInputs()));
    connect(fromLongitudeLineEdit, SIGNAL(textChanged(const QString &)),
            this, SLOT(checkLineEditInputs()));
    connect(toLongitudeLineEdit, SIGNAL(textChanged(const QString &)),
            this, SLOT(checkLineEditInputs()));

    setAttribute(Qt::WA_DeleteOnClose, true);

}

//! Destructor
MapTileDownloadDialog::~MapTileDownloadDialog()
{
}

//! Returns the zoomlevel selected in dialog
int MapTileDownloadDialog::selectedZoomLevel()
{
    return zoomLevelSpinBox->value();
}

//! returns the fromLatitude as qreal
qreal MapTileDownloadDialog::fromLatitude()
{
    return fromLatitudeLineEdit->text().toDouble();
}

//! returns the toLatitude as qreal
qreal MapTileDownloadDialog::toLatitude()
{
    return toLatitudeLineEdit->text().toDouble();
}

//! returns the fromLongitude as qreal
qreal MapTileDownloadDialog::fromLongitude()
{
    return fromLongitudeLineEdit->text().toDouble();
}

//! returns the toLongitude as qreal
qreal MapTileDownloadDialog::toLongitude()
{
    return toLongitudeLineEdit->text().toDouble();
}

//! Returns a pointer to "Select area" button in dialog
QPushButton *MapTileDownloadDialog::getSelectAreaButton()
{
    return selectAreaButton;
}

//! Sets the value given as parameter to fromLatitudeLineEdit
void MapTileDownloadDialog::setFromLatitude(qreal latitude)
{
    fromLatitudeLineEdit->setText(tr("%1").arg(latitude));
}

//! Sets the value given as parameter to tLatitudeLineEdit
void MapTileDownloadDialog::setToLatitude(qreal latitude)
{
    toLatitudeLineEdit->setText(tr("%1").arg(latitude));
}

//! Sets the value given as parameter to fromLongitudeLineEdit
void MapTileDownloadDialog::setFromLongitude(qreal longitude)
{
    fromLongitudeLineEdit->setText(tr("%1").arg(longitude));
}

//! Sets the value given as parameter to toLongitudeLineEdit
void MapTileDownloadDialog::setToLongitude(qreal longitude)
{
    toLongitudeLineEdit->setText(tr("%1").arg(longitude));
}

//! Checks inputs and if they are ok, enables the Ok button
void MapTileDownloadDialog::checkLineEditInputs()
{

    if (fromLatitudeLineEdit->hasAcceptableInput() &&
        toLatitudeLineEdit->hasAcceptableInput() &&
        fromLongitudeLineEdit->hasAcceptableInput() &&
        toLongitudeLineEdit->hasAcceptableInput()) {
            
        buttonBox->button(QDialogButtonBox::Ok)->setEnabled(true);
    }
    else {
        buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
    }
}
