/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __MAPTILE_H__
#define __MAPTILE_H__

#include "maprepository.h"

#include <QGraphicsPixmapItem>

class QString;

/*!
 * Class MapTile represents a map tile.
 */
class MapTile : public QGraphicsPixmapItem
{
public:
    //! Default constructor.
    MapTile(QGraphicsItem *parent = 0);

    /*!
     * Constructs a MapTile object with given values.
     *
     * \param repository The repository from which the map tile is from.
     * \param parent The parent item.
     */
    MapTile(MapRepository repository, QGraphicsItem *parent = 0);

    /*!
     * Constructs a MapTile object with given values.
     *
     * \param repository The repository from which the map tile is from.
     * \param zoomLevel The zoom level of the map tile.
     * \param mapXCoordinate The x coordinate of the map tile.
     * \param mapYCoordinate The y coordinate of the map tile.
     * \param parent The parent item.
     */
    MapTile(MapRepository repository, quint32 zoomLevel,
            quint32 mapXCoordinate, quint32 mapYCoordinate,
            QGraphicsItem *parent = 0);

    /*!
     * Constructs a MapTile object with given values.
     *
     * \param repository The repository from which the map tile is from.
     * \param zoomLevel The zoom level of the map tile.
     * \param mapXCoordinate The x coordinate of the map tile.
     * \param mapYCoordinate The y coordinate of the map tile.
     * \param screenXCoordinate The screen x coordinate of the tile.
     * \param screenYCoordinate,The screen y coordinate of the tile.
     * \param pixmap The map image.
     * \param parent The parent item.
     */
    MapTile(MapRepository repository, quint32 zoomLevel,
            quint32 mapXCoordinate, quint32 mapYCoordinate,
            quint32 screenXCoordinate, quint32 screenYCoordinate,
            QPixmap pixmap, QGraphicsItem *parent = 0);

    //! The lowest permitted type value for custom items. Used with type().
    enum { Type = UserType + 1 };

    /*!
     * \brief Return the type of this item.
     *
     * This method is needed to enable use of the qgraphicsitem_cast().
     *
     * \return The type of this item as an int.
     */
    virtual int type() const { return Type; }

    //! Returns the repository property.
    MapRepository getRepository() const { return repository; }

    //! Sets the repository property.
    void setRepository(MapRepository repository) { this->repository =
                                                   repository; }

    //! Returns the x-coordinate property.
    qint32 getMapXCoordinate() const { return mapXCoordinate; }

    //! Sets the x-coordinate property.
    void setMapXCoordinate(qint32 mapXCoordinate) { this->mapXCoordinate =
                                                    mapXCoordinate; }

    //! Returns the y-coordinate property.
    qint32 getMapYCoordinate() const { return mapYCoordinate; }

    //! Sets the y-coordinate property.
    void setMapYCoordinate(qint32 mapYCoordinate) { this->mapYCoordinate =
                                                    mapYCoordinate; }

    //! Returns the zoom level property.
    qint32 getZoomLevel() const { return zoomLevel; }

    //! Sets the zoom level property.
    void setZoomlevel(qint32 zoomLevel) { this->zoomLevel = zoomLevel; }

    //! Returns the key property.
    QString getKey() const { return key; }

    //! Returns the screenXCoordinate property.
    qint32 getScreenXCoordinate() const { return screenXCoordinate; }

    //! Sets the screenXCoordinate property.
    void setScreenXCoordinate(qint32 screenXCoordinate) {
        this->screenXCoordinate = screenXCoordinate; }

    //! Returns the mapYCoordinate property.
    qint32 getScreenYCoordinate() const { return screenYCoordinate; }

    //! Sets the screenYCoordinate property.
    void setScreenYCoordinate(qint32 screenYCoordinate) {
        this->screenYCoordinate = screenYCoordinate; }

    /*!
     * \brief Generates a key string.
     *
     * In order to store and retrieve map images to and from the
     * image cache a unique key string is needed. This function generates
     * a suitable key using the mapXCoordinate, mapYCoordinate
     * and zoomLevel properties.
     *
     * \param zoomLevel The zoom level of the map tile.
     * \param mapXCoordinate The x coordinate of the map tile.
     * \param mapYCoordinate The y coordinate of the map tile.
     * \return The created key string.
     */
    QString createKey(quint32 zoomLevel, quint32 mapXCoordinate,
                      quint32 mapYCoordinate);

private:
    //! The map repository from which the map image is from.
    MapRepository repository;

    //! The zoom level of the map tile.
    qint32 zoomLevel;

    //! The x-coordinate of the map tile.
    qint32 mapXCoordinate;

    //! The y-coordinate of the map tile.
    qint32 mapYCoordinate;

    //! A string used to identify the image in the cache.
    QString key;

    //! The x coordinate where the map tile is drawn on screen.
    qint32 screenXCoordinate;

    //! The x coordinate where the map tile is drawn on screen.
    qint32 screenYCoordinate;
};

#endif 
