/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __MAPDOWNLOADER_H__
#define __MAPDOWNLOADER_H__

#include <QObject>
#include <QPixmap>
#include <QPointer>
#include "maprepository.h"

class QHttp;
class MapTile;

/*!
 * Download map images from an Internet map repository.
 */
class MapDownloader : public QObject
{
    Q_OBJECT

private:
    //! The "http://" prexix in the url.
    const QString urlHttpPrefix;

    //! An empty string.
    const QString emptyString;

    //! Used for a HTTP connection.
    QHttp *http;

    //! ID number of a HTTP request.
    int requestId;

    //! Repository of the image to download.
    MapRepository m_repository;

    //! Zoomlevel of the image to download.
    quint32 m_zoom;

    //! Map x coordinate of the image to download.
    quint32 m_mapX;

    //! Map y coordinate of the image to download.
    quint32 m_mapY;

    //! Screen x coordinate of the image to download.
    quint32 m_screenX;

    //! Screen y coordinate of the image to download.
    quint32 m_screenY;

    //! The downloaded image.
    QPixmap m_pixmap;

    //! Reset the image data.
    void reset();

public:
    //! Constructor.
    MapDownloader();

    //! Destructor.
    ~MapDownloader();

    /*!
     * \brief Downloads a map tile.
     *
     * Downloads a map tile from the specified repository and coordinates.
     */
    void downloadMapTile(MapRepository repository,
                         quint32 zoomLevel,
                         quint32 mapXCoordinate,
                         quint32 mapYCoordinate,
                         qint32 screenXCoordinate,
                         qint32 screenYCoordinate);

    /*!
     * Get the downloaded image.
     *
     * \return The downloaded image.
     */
    MapTile *getDownloadedImage();

    //! Cancel a download in progress.
    void abortDownload();

private slots:

    /*!
     * \brief Read image data after download is finished.
     *
     * Connected to the requestFinished signal of the http object.
     * When a download is finished, this function reads the image data
     * and saves it into the downloadedImage member. The imageDownloaded
     * signal is sent in case of a successful download, or if an error
     * occurred, the imageDownloadFailed signal is sent.
     *
     * \param id The HTTP request ID.
     * \param error True if a HTTP error occurred, false otherwise.
     */
    void readImageData(int id, bool error);

    //! Connected to the http object's done() signal.
    void requestsFinished();

    //! Connected to the http object's requestStarted() signal.
    void requestStarted();

signals:

    //! This signal is sent after a successful download.
    void imageDownloaded();

    /*!
     * This signal is sent if an error occurred during download.
     *
     * \param error A message describing the error.
     */
    void imageDownloadFailed(QString error);

    //! Emitted when there is a new http request.
    void httpRequestsPending();

    //! Emitted when all http requests are complete.
    void allHttpRequestsComplete();
};

#endif
