/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QHttp>
#include <QPixmap>
#include <QString>
#include "maprepository.h"
#include "maptile.h"
#include "mapdownloader.h"

//! Constructor.
MapDownloader::MapDownloader() :
    urlHttpPrefix("http://"),
    emptyString("")
{
    http = new QHttp();
    connect(http, SIGNAL(requestFinished(int, bool)), this,
            SLOT(readImageData(int, bool)));
    connect(http, SIGNAL(done(bool)), this, SLOT(requestsFinished()));
    connect(http, SIGNAL(requestStarted(int)), this, SLOT(requestStarted()));

    requestId = 0;
}

//! Destructor.
MapDownloader::~MapDownloader()
{
    delete http;
}

//! Downloads a map tile.
void MapDownloader::downloadMapTile(MapRepository repository,
                                    quint32 zoomLevel,
                                    quint32 mapXCoordinate,
                                    quint32 mapYCoordinate,
                                    qint32 screenXCoordinate,
                                    qint32 screenYCoordinate)
{
    const QString fileExtension(".png");
    QString fileName;

    reset();

    fileName.append(repository.getImageNameFormat()
                    .arg(zoomLevel)
                    .arg(mapXCoordinate)
                    .arg(mapYCoordinate));
    fileName.append(fileExtension);

    m_repository = repository;
    m_zoom = zoomLevel;
    m_mapX = mapXCoordinate;
    m_mapY = mapYCoordinate;
    m_screenX = screenXCoordinate;
    m_screenY = screenYCoordinate;

    // Replace the "http://" prefix in the address if there is one.
    http->setHost(repository.getUrl().replace(urlHttpPrefix, emptyString));
    requestId = http->get(fileName);
}

//! Read image data after download is finished.
void MapDownloader::readImageData(int id, bool error)
{
    QByteArray data;

    if (id == requestId) {
        // In case of HTTP error, send a signal that the download has failed.
        if (error) {
            reset();
            emit imageDownloadFailed(http->errorString());
            return;
        }

        // Read all pending data from the http object.
        data = http->readAll();

        // Create an image from the data.
        m_pixmap.loadFromData(data);

        /* If the downloaded data is valid image data,
         * send a signal that the image is ready to be used.
         * Otherwise send a signal that the image download has failed.
         */
        if (!data.isNull()) {
            requestId = 0;
            emit imageDownloaded();
        } 
        else {
            reset();
            emit imageDownloadFailed(tr("Invalid image."));
        }
    }
}

//! Get the downloaded image.
MapTile *MapDownloader::getDownloadedImage()
{
    MapTile *tile = new MapTile(m_repository, m_zoom, m_mapX, m_mapY,
                                m_screenX, m_screenY, m_pixmap);
    reset();
    return tile;
}

//! Cancel a download in progress.
void MapDownloader::abortDownload()
{
    reset();

    // Clear all pending HTTP requests.
    http->clearPendingRequests();

    // Abort a request in progress if there is one.
    http->abort();
}

//! Connected to the http object's done() signal.
void MapDownloader::requestsFinished()
{
    emit allHttpRequestsComplete();
}

//! Connected to the http object's requestStarted() signal.
void MapDownloader::requestStarted()
{
    emit httpRequestsPending();
}

//! Reset the image data.
void MapDownloader::reset()
{
    m_repository = MapRepository();
    m_zoom = 0;
    m_mapX = 0;
    m_mapY = 0;
    m_screenX = 0;
    m_screenY = 0;
    m_pixmap = QPixmap();
}
