/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __MAPDATABASE_H__
#define __MAPDATABASE_H__

#include <QSqlDatabase>
#include <QString>

class QPixmap;

#include "maptile.h"

/*!
 * A local database to store the map information into.
 */
class MapDatabase
{
public:

    /*!
     * \brief Constructor.
     *
     * Constructs a MapDatabase object and initializes it with given values
     * and creates a new database if it doesn't already exist.
     *
     * \param repository The name of the database file is determined by the
     * database member of a MapRepository object.
     * \param aMapimageDirectory The name of the directory to save the map
     * images into.
     */
    MapDatabase(MapRepository repository, const QString &aMapImageDirectory);

    //! Destructor.
    ~MapDatabase();

    /*!
     * Creates database connection.
     *
     * \return True if database was opened successfully, false otherwise.
     */
    bool createConnection();

    /*!
     * Inserts a map tile to database and saves the image of the map tile.
     *
     * \param tile The tile to insert.
     * \return True if the insertion was successful, false otherwise.
     */
    bool insertMapTile(MapTile *tile);

    /*!
     * \brief Replace map tile in database.
     * Replace a map tile to created database if it already exists
     * and saves the image of the map tile.
     *
     * \param tile The tile to replace.
     * \return True if replacement was successful, false otherwise.
     */
    bool replaceMapTile(MapTile *tile);

    /*!
     * \brief Remove tile from database.
     * Removes a map tile from created database and the
     * image of the map tile from file.
     *
     * \param key The key of the tile.
     * \return True if removal was successful, false otherwise.
     */
    bool removeMapTile(const QString &key);

    /*!
     * Find a map tile in the database.
     *
     * \param key The key of the tile.
     * \return True if the tile was found, false otherwise.
     */
    bool findMapTile(const QString &key);

    /*!
     * Returns a map tile matching to the key from database.
     *
     * \param key The key of the tile.
     * \return The new map tile.
     */
    MapTile * getMapTile(const QString &key);

    //! Closes a database connection.
    void closeConnection() {
        this->db.close();
    }

    /*!
     * If database and map images are moved to a new directory, call this
     *
     * \param newImageDirectory The new map directory name.
     */
    void changeDatabaseDirectory(const QString &newImageDirectory);

    //! Prints the contents of the database. For testing purposes.
    void printDatabaseInformation();

    /*!
     * Get the state of the database connection (connected/not connected).
     *
     * \return True if the database connection is open, false otherwise.
     */
    bool connected() {
        return this->db.isOpen();
    }

private:
    //! Database type.
    const QString dbType;

    //! Name for the database connection.
    const QString connectionName;

    //! Creates a table of database for map tiles.
    void createTable();

    //! for handling database
    QSqlDatabase db;

    //! for storing the map image directory
    QString mapImageDirectory;

};

#endif
